#ifndef TOKEN_MANAGER_H
#define TOKEN_MANAGER_H

#include "common/token/token.h"

// 第一位是 NORMAL
// 第二位是 REVERSED
#define DIR_NORMAL 1
#define DIR_REVERSED 2
#define DIR_EMPTY 0
#define MAKE_EMPTY(state) ((state) << 2)
#define MAKE_NORMAL(state) (((state) << 2) | 1)
#define MAKE_REVERSED(state) (((state) << 2) | 2)
#define SET_REVERSED(state) ((state) = ((state) | 2) & ~1)
#define SET_NORMAL(state) ((state) = ((state) | 1) & ~2)
#define GET_STATE(state) ((state) >> 2)
#define IS_EDGE_REVERSED(state) (( state )&2)
#define IS_EDGE_NORMAL(state) (( state )&1)
#define STATE_DIRECTION(state) (( state )&3)

#define EMPTY_STATE_COUNT 16
#define STATE_VARIABLE_BITS 4
#define IS_EMPTY_STATE_INTERNAL(state) \
    (((state) >> STATE_VARIABLE_BITS) < EMPTY_STATE_COUNT)
#define IS_EMPTY_STATE(directed_state) \
    IS_EMPTY_STATE_INTERNAL(GET_STATE(directed_state))
#define STATE_INDEX_INTERNAL(state) ((state) >> STATE_VARIABLE_BITS)
#define STATE_INDEX(directed_state) \
    STATE_INDEX_INTERNAL(GET_STATE(directed_state))
#define STATE_VAR_COUNT_INTERNAL(state) (( state )&0xf)
#define STATE_VAR_COUNT(directed_state) \
    STATE_VAR_COUNT_INTERNAL(GET_STATE(directed_state))

#define SENTENCE_VAR -16
#define LEMMA_VAR -1
#define MAKE_VAR(major, minor) (((major) << STATE_VARIABLE_BITS) | (minor))
#define VAR_MAJOR(var) STATE_INDEX_INTERNAL(var)
#define VAR_MINOR(var) STATE_VAR_COUNT_INTERNAL(var)
#define IS_SENTENCE_VAR(var) ((var) == SENTENCE_VAR)
#define IS_LITERAL_VAR(var) ((var) <= -32)
#define IS_SENSE_VAR(var) ((var) == -3)
#define IS_THAT_VAR(var) ((var) == -2)
#define IS_LEMMA_VAR(var) ((var) == LEMMA_VAR)
#define IS_NORMAL_VAR(var) ((var) >= 0)
#define VAR_TO_STRING(var) \
    ((( char )(VAR_MAJOR(var) + 97)) + std::to_string(VAR_MINOR(var)))

#define THAT_VAR "<THAT>"

#define LITERAL_INDEX(literal) (-( literal )-32)

#define NULL_POSTAG '#'
#define NULL_SENSE "#"
#define NULL_LEMMA "#"
#define NULL_CARG "#"

class TokenManager {

protected:
    static Token lemmas_;
    static Token node_labels_;
    static Token senses_;
    static Token edge_labels_;

    // load manually
    static std::vector< ttoken > states_;
    static std::unordered_map< std::string, int > states_to_indices_;
    static std::unordered_map< std::string, std::vector< int > > states_map_;
    static std::vector< int > states_appropriate_var_count_;
    static std::vector< ttoken > literals_;

    static bool loadTokens(const std::string& file_name,
                           std::vector< ttoken >& tokens);

public:
    static int indexOfNodeLabel(const ttoken& node_label);
    static const ttoken& nodeLabelAt(int node_label_index);
    static int nodeLabelCount();
    static Token& nodeLabels();

    static int indexOfEdgeLabel(const ttoken& edge_label);
    static const ttoken& edgeLabelAt(int edge_label_index);
    static int edgeLabelCount();
    static Token& edgeLabels();

    static int indexOfLemma(const ttoken& lemma);
    static const ttoken& lemmaAt(int lemma_index);
    static int lemmaCount();
    static Token& lemmas();

    static int indexOfSense(const ttoken& sense);
    static const ttoken& senseAt(int sense_index);
    static int senseCount();
    static Token& senses();

    static const ttoken& literalAt(int literal_index);
    static bool loadLiterals(const std::string& literal_file);
    static int literalCount();

    static const std::vector< ttoken >& states();
    static const ttoken& stateAt(int state_index);
    static bool loadStates(const std::string& state_file);
    static int stateCount();

    static int indexOfState(const ttoken& state);
    static int makeState(int state_index, bool reversed);
    static const std::vector< int >& statesOfEdge(const ttoken& edge_label);
    static bool stateMatch(int state_index, const ttoken& node_label);
};

inline int TokenManager::indexOfNodeLabel(const ttoken& node_label) {
    return node_labels_.lookup(node_label);
}
inline const ttoken& TokenManager::nodeLabelAt(int node_label_index) {
    return node_labels_.key(node_label_index);
}
inline int TokenManager::nodeLabelCount() {
    return node_labels_.count();
}
inline Token& TokenManager::nodeLabels() {
    return node_labels_;
}

inline int TokenManager::indexOfEdgeLabel(const ttoken& edge_label) {
    return edge_labels_.lookup(edge_label);
}
inline const ttoken& TokenManager::edgeLabelAt(int edge_label_index) {
    return edge_labels_.key(edge_label_index);
}
inline int TokenManager::edgeLabelCount() {
    return edge_labels_.count();
}
inline Token& TokenManager::edgeLabels() {
    return edge_labels_;
}

inline int TokenManager::indexOfLemma(const ttoken& lemma) {
    return lemmas_.lookup(lemma);
}
inline const ttoken& TokenManager::lemmaAt(int lemma_index) {
    return lemmas_.key(lemma_index);
}
inline int TokenManager::lemmaCount() {
    return lemmas_.count();
}
inline Token& TokenManager::lemmas() {
    return lemmas_;
}

inline int TokenManager::indexOfSense(const ttoken& sense) {
    return senses_.lookup(sense);
}
inline const ttoken& TokenManager::senseAt(int sense_index) {
    return senses_.key(sense_index);
}
inline int TokenManager::senseCount() {
    return senses_.count();
}
inline Token& TokenManager::senses() {
    return senses_;
}

inline const ttoken& TokenManager::literalAt(int literal) {
    return literals_[LITERAL_INDEX(literal)];
}
inline int TokenManager::literalCount() {
    return literals_.size();
}
inline const ttoken& TokenManager::stateAt(int directed_state) {
    return states_[STATE_INDEX(directed_state)];
}
inline int TokenManager::stateCount() {
    return states_.size();
}

inline const std::vector< ttoken >& TokenManager::states() {
    return states_;
}

inline bool TokenManager::loadLiterals(const std::string& literal_file) {
    return loadTokens(literal_file, literals_);
}

inline int TokenManager::makeState(int state_index, bool reversed) {
    if (state_index < EMPTY_STATE_COUNT)
        return MAKE_EMPTY(state_index << STATE_VARIABLE_BITS);
    int state = (state_index << STATE_VARIABLE_BITS) +
                states_appropriate_var_count_[state_index];
    return reversed ? MAKE_REVERSED(state) : MAKE_NORMAL(state);
}

inline const std::vector< int >&
TokenManager::statesOfEdge(const ttoken& edge_label) {
    static std::vector< int > empty;
    auto iter = states_map_.find(edge_label);
    if (iter == states_map_.end())
        return empty;
    return iter->second;
}


#endif /* TOKEN_MANAGER_H */

// Local Variables:
// mode: c++
// End:
