#!/usr/bin/env python
# coding=utf-8
# Copyright 2021 The HuggingFace Team All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""
Fine-tuning the library's seq2seq models for question answering using the 🤗 Seq2SeqTrainer.
"""
# You can also adapt this script on your own question answering task. Pointers for this are left as comments.

import logging
import os
import sys
import random
from dataclasses import dataclass, field
from typing import List, Optional, Tuple
from tqdm import tqdm
from collections import Counter

import torch
import datasets
from datasets import load_dataset, DatasetDict, concatenate_datasets, load_from_disk
import numpy as np
import pandas as pd
import transformers
from transformers import (
    AutoConfig,
    AutoModelForSeq2SeqLM,
    AutoTokenizer,
    DataCollatorForSeq2Seq,
    HfArgumentParser,
    set_seed
)
from transformers.trainer_utils import get_last_checkpoint
from transformers.utils import check_min_version
from peft import PeftModel, PeftConfig
from peft import get_peft_config, get_peft_model, get_peft_model_state_dict, LoraConfig, TaskType

from src.utils import ModelArguments, DataTrainingArguments, Seq2SeqArguments, LoraArguments, TASK_DEMONSTRATION_ONLY, get_dataset_sizes
from src.metrics import Bleu
from src.models import HyperLoRAConfig, HyperLoRAModelForPretrain
from src.trainer import Seq2SeqHyperTrainer

import os
os.environ["WANDB_PROJECT"]="HyperLoRA"

logger = logging.getLogger(__name__)

def main():
    # See all possible arguments in src/transformers/training_args.py
    # or by passing the --help flag to this script.
    # We now keep distinct sets of args, for a cleaner separation of concerns.

    parser = HfArgumentParser((ModelArguments, DataTrainingArguments, Seq2SeqArguments, LoraArguments))
    if len(sys.argv) == 2 and sys.argv[1].endswith(".json"):
        # If we pass only one argument to the script and it's the path to a json file,
        # let's parse it to get our arguments.
        model_args, data_args, training_args, lora_args = parser.parse_json_file(json_file=os.path.abspath(sys.argv[1]))
    else:
        model_args, data_args, training_args, lora_args = parser.parse_args_into_dataclasses()

    # update model_args

    # Setup logging
    logging.basicConfig(
        format="%(asctime)s - %(levelname)s - %(name)s - %(message)s",
        datefmt="%m/%d/%Y %H:%M:%S",
        handlers=[logging.StreamHandler(sys.stdout)],
    )

    log_level = training_args.get_process_log_level()
    logger.setLevel(log_level)
    datasets.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.enable_default_handler()
    transformers.utils.logging.enable_explicit_format()

    training_args.local_rank = -1
    # Log on each process the small summary:
    logger.warning(
        f"Process rank: {training_args.local_rank}, device: {training_args.device}, n_gpu: {training_args.n_gpu}"
        + f", distributed training: {bool(training_args.local_rank != -1)}, 16-bits training: {training_args.fp16}"
    )
    logger.info(f"Training/evaluation parameters {training_args}")

    # Set seed before initializing model.
    set_seed(training_args.seed)

    # read multi-task datasets
    if 'merged' not in data_args.dataset_name:
        train_dataset_sizes, val_dataset_sizes = [], []
        raw_train, raw_validation = [], []
        for task_file in tqdm(os.listdir(data_args.dataset_name), desc='Loading Datasets', total=len(os.listdir(data_args.dataset_name))):
            task_dataset = load_dataset("json", data_files=os.path.join(data_args.dataset_name, task_file))
            # random sample n demonstrations
            rnd_demo_idx = random.sample(range(len(task_dataset["train"])), data_args.n_demonstrations)
            task_dataset['train'] = task_dataset['train'].filter(lambda x: x not in rnd_demo_idx)
            # intra-task shuffle
            task_dataset['train'] = task_dataset["train"].shuffle()
            task_dataset = task_dataset["train"].train_test_split(test_size=0.2)
            task_dataset["validation"] = task_dataset["test"]

            train_dataset_sizes.append(len(task_dataset['train']))
            val_dataset_sizes.append(len(task_dataset['validation']))

            raw_train.append(task_dataset["train"])
            raw_validation.append(task_dataset["validation"])

        raw_datasets = DatasetDict({"train": concatenate_datasets(raw_train),
                                    "validation": concatenate_datasets(raw_validation)})
    else:
        raw_datasets = load_from_disk(data_args.dataset_name)
        train_dataset_sizes = get_dataset_sizes(raw_datasets, split='train')
        val_dataset_sizes = get_dataset_sizes(raw_datasets, split='validation')

    # Load pretrained model and tokenizer
    #
    # Distributed training:
    # The .from_pretrained methods guarantee that only one local process can concurrently
    # download model & vocab.
    config = AutoConfig.from_pretrained(
        model_args.config_name if model_args.config_name else model_args.model_name_or_path,
        cache_dir=model_args.cache_dir,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )
    tokenizer = AutoTokenizer.from_pretrained(
        model_args.tokenizer_name if model_args.tokenizer_name else model_args.model_name_or_path,
        cache_dir=model_args.cache_dir,
        use_fast=model_args.use_fast_tokenizer,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )
    model = AutoModelForSeq2SeqLM.from_pretrained(
        model_args.model_name_or_path,
        from_tf=bool(".ckpt" in model_args.model_name_or_path),
        config=config,
        cache_dir=model_args.cache_dir,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )

    def get_task_lora_path(task_name):
        # if ':' in task_name:
        #     task_name = task_name.split(':')[0]
        task_name = task_name.replace('/', '_')
        # model_path = f'flan_t5_base-{task_name}'
        # TODO:
        model_path = f'flan_t5_base-{task_name}.json/'

        return model_path

    # evaluate based on task_names
    org_output_dir = training_args.output_dir
    task_names = list(set(raw_datasets['validation']['task_name']))
    task2eval_metric = pd.DataFrame(columns=['task_name', 'exact_match', 'bleu', 'nums'])
    for task_name in tqdm(task_names, total=len(task_names), desc='Evaluating on each task'):
        # output
        training_args.output_dir = os.path.join(org_output_dir, task_name)
        if os.path.exists(os.path.join(training_args.output_dir, 'all_results.json')):
            continue

        eval_dataset = raw_datasets['validation'].filter(lambda example: example['task_name'] == task_name)
        val_dataset_sizes = [len(eval_dataset)]
        model_path = get_task_lora_path(task_name)
        model = PeftModel.from_pretrained(model, model_path)

        # We resize the embeddings only when necessary to avoid index errors. If you are creating a model from scratch
        # on a small vocab and want a smaller embedding size, remove this test.
        embedding_size = model.get_input_embeddings().weight.shape[0]
        if len(tokenizer) > embedding_size:
            model.resize_token_embeddings(len(tokenizer))

        if model.config.decoder_start_token_id is None:
            raise ValueError("Make sure that `config.decoder_start_token_id` is correctly defined")

        column_names = eval_dataset.column_names
        input_column = data_args.input_column
        output_column = data_args.output_column

        # Temporarily set max_answer_length for training.
        max_answer_length = data_args.max_answer_length
        padding = "max_length" if data_args.pad_to_max_length else False

        if training_args.label_smoothing_factor > 0 and not hasattr(model, "prepare_decoder_input_ids_from_labels"):
            logger.warning(
                "label_smoothing is enabled but the `prepare_decoder_input_ids_from_labels` method is not defined for"
                f"`{model.__class__.__name__}`. This will lead to loss being calculated twice and will take up more memory"
            )

        max_seq_length = data_args.max_seq_length

        def preprocess_function(examples):
            # impl: random sample demonstrations
            inputs = examples[input_column]
            targets = examples[output_column]
            # task_name = examples['task_name']

            model_inputs = tokenizer(inputs, max_length=max_seq_length, padding=padding, truncation=True)
            # Tokenize targets with text_target=...
            labels = tokenizer(text_target=targets, max_length=max_answer_length, padding=padding, truncation=True)

            # padding in the loss.
            if padding == "max_length" and data_args.ignore_pad_token_for_loss:
                labels["input_ids"] = [
                    [(l if l != tokenizer.pad_token_id else -100) for l in label] for label in labels["input_ids"]
                ]

            model_inputs["labels"] = labels["input_ids"]
            # model_inputs['task_name'] = task_name
            return model_inputs

        if data_args.max_eval_samples is not None:
            # We will select sample from whole data
            max_eval_samples = min(len(eval_dataset), data_args.max_eval_samples)
            eval_dataset = eval_dataset.select(range(max_eval_samples))
        # Validation Feature Creation
        with training_args.main_process_first(desc="validation dataset map pre-processing"):
            eval_dataset = eval_dataset.map(
                preprocess_function,
                batched=True,
                num_proc=data_args.preprocessing_num_workers,
                remove_columns=column_names,
                load_from_cache_file=not data_args.overwrite_cache,
                desc="Running tokenizer on validation dataset",
            )
        if data_args.max_eval_samples is not None:
            # During Feature creation dataset samples might increase, we will select required samples again
            max_eval_samples = min(len(eval_dataset), data_args.max_eval_samples)
            eval_dataset = eval_dataset.select(range(max_eval_samples))

        # Data collator
        label_pad_token_id = -100 if data_args.ignore_pad_token_for_loss else tokenizer.pad_token_id
        data_collator = DataCollatorForSeq2Seq(
            tokenizer,
            model=model,
            label_pad_token_id=label_pad_token_id,
            pad_to_multiple_of=8 if training_args.fp16 else None
        )

        bleu_eval = Bleu()

        def get_exact_match(predictions: List[str], references: List[str], **kwargs):
            """
            Exact match as the default evaluation
            """
            assert len(predictions) == len(references)
            correct_num = 0
            for prediction, reference in zip(predictions, references):
                if prediction.lower() == reference.lower():
                    correct_num += 1
            return correct_num / len(predictions)

        def compute_metrics(eval_preds):
            preds, labels = eval_preds
            if isinstance(preds, tuple):
                preds = preds[0]

            if data_args.ignore_pad_token_for_loss:
                # Replace -100 in the labels as we can't decode them.
                labels = np.where(labels != -100, labels, tokenizer.pad_token_id)
                preds = np.where(preds != -100, preds, tokenizer.pad_token_id)

            decoded_preds = tokenizer.batch_decode(preds, skip_special_tokens=True)
            decoded_labels = tokenizer.batch_decode(labels, skip_special_tokens=True)

            # Some simple post-processing
            decoded_preds, decoded_labels = postprocess_text(decoded_preds, decoded_labels)
            result = get_exact_match(predictions=decoded_preds, references=decoded_labels)
            bleu_result = bleu_eval.compute(predictions=decoded_preds, references=decoded_labels)
            return {"exact_match": result, "bleu": bleu_result["bleu"]}

        # Post-processing:
        def postprocess_text(preds, labels):
            preds = [pred.strip() for pred in preds]
            labels = [label.strip() for label in labels]
            return preds, labels

        # no shuffle in inter-task
        # train_dataset = train_dataset.shuffle(seed=training_args.seed)
        # call_backs = [EarlyStoppingCallback(early_stopping_patience=3)]
        # Initialize our Trainer
        trainer = Seq2SeqHyperTrainer(
            train_dataset_sizes=train_dataset_sizes,
            val_dataset_sizes=val_dataset_sizes,
            model=model,
            args=training_args,
            eval_dataset=eval_dataset if training_args.do_eval else None,
            tokenizer=tokenizer,
            data_collator=data_collator,
            compute_metrics=compute_metrics if training_args.predict_with_generate else None,
            # callbacks=call_backs
        )

        # Evaluation
        max_length = (
            training_args.generation_max_length
            if training_args.generation_max_length is not None
            else data_args.val_max_answer_length
        )
        num_beams = data_args.num_beams if data_args.num_beams is not None else training_args.generation_num_beams
        logger.info("*** Evaluate ***")
        metrics = trainer.evaluate(max_length=max_length, num_beams=num_beams, metric_key_prefix="eval")

        max_eval_samples = data_args.max_eval_samples if data_args.max_eval_samples is not None else len(eval_dataset)
        metrics["eval_samples"] = min(max_eval_samples, len(eval_dataset))

        trainer.log_metrics("eval", metrics)
        trainer.save_metrics("eval", metrics)

        task2eval_metric.loc[len(task2eval_metric.index)] = [task_name, metrics['eval_exact_match'], metrics['eval_bleu'], len(eval_dataset)]

    task2eval_metric.to_csv(os.path.join(org_output_dir, 'task2eval_metric.csv'))

def _mp_fn(index):
    # For xla_spawn (TPUs)
    main()


if __name__ == "__main__":
    main()
