///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// This file is part of ModelBlocks. Copyright 2009, ModelBlocks developers. //
//                                                                           //
//    ModelBlocks is free software: you can redistribute it and/or modify    //
//    it under the terms of the GNU General Public License as published by   //
//    the Free Software Foundation, either version 3 of the License, or      //
//    (at your option) any later version.                                    //
//                                                                           //
//    ModelBlocks is distributed in the hope that it will be useful,         //
//    but WITHOUT ANY WARRANTY; without even the implied warranty of         //
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          //
//    GNU General Public License for more details.                           //
//                                                                           //
//    You should have received a copy of the GNU General Public License      //
//    along with ModelBlocks.  If not, see <http://www.gnu.org/licenses/>.   //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <getopt.h>

#include "nl-iomacros.h"
#include "nl-cpt.h"



char psX[]   = "";
char psSpc[] = " ";


//// G: grammar rule symbol
DiscreteDomain<int> domG;
typedef DiscreteDomainRV<int,domG> G;
const G G_NONE ("-");

//// GG: pair of grammar rule symbols associated with rule expansion
typedef DelimitedJoint2DRV<psX,G,psSpc,G,psX> GG;
const GG GG_NONE ( G_NONE, G_NONE );

//// Model of G (prior)
typedef CPT1DModel<G,Prob> GModel;
//// Model of GG given G (binary branching expansion)
typedef CPT2DModel<GG,G,Prob> GGModel;


// //// PW: pos#word (terminal symbol)
// DiscreteDomain<int> domPW;
// typedef DiscreteDomainRV<int,domPW> PW;

//// P: pos symbol
DiscreteDomain<int> domP;
typedef DiscreteDomainRV<int,domP> P;
const P P_NONE ("-");

//// W: pos symbol
DiscreteDomain<int> domW;
typedef DiscreteDomainRV<int,domW> W;
const W W_NONE ("-");

//// Model of term P given G
typedef CPT2DModel<P,G,Prob> PgModel;
//// Model of term P given W
typedef CPT2DModel<P,W,Prob> PwModel;
//// Model of term P
typedef CPT1DModel<P,Prob> PModel;
//// Model of word W
typedef CPT1DModel<W,Prob> WModel;


//// B: boolean
DiscreteDomain<int> domB;
typedef DiscreteDomainRV<int,domB> B;
const B B_0 ("0");
const B B_1 ("1");

//// C: right-corner transformed grammar rule symbol
DiscreteDomain<int> domC;
typedef DiscreteDomainRV<int,domC> C;
const C C_NONE ("-");

//// CC: pair of right-corner transformed grammar rule symbols associated with rule expansion
typedef DelimitedJoint2DRV<psX,C,psSpc,C,psX> CC;
const CC CC_NONE ( C_NONE, C_NONE );

//// Model of CC given C (binary branching expansion)
typedef CPT2DModel<CC,C,Prob> CCModel;
//// Model of C given C (unary branching expansion)
typedef CPT2DModel<C,C,Prob> CuModel;


//// D: depth (input only, to HHMM models)...
DiscreteDomain<char> domK;
typedef DiscreteDomainRV<char,domK> K;
const K K_00("00");
const K K_01("01");
const K K_02("02");
const K K_03("03");
const K K_04("04");
const K K_05("05");
const K K_06("06");
const K K_07("07");
const K K_08("08");
const K K_09("09");
const K K_10("10");
const K K_11("11");
const K K_12("12");
const K K_13("13");
const K K_14("14");
const K K_15("15");
const K K_16("16");
const K K_17("17");
const K K_18("18");
const K K_19("19");
const K K_20("20");


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
///
///  Main function
///
////////////////////////////////////////////////////////////////////////////////

static struct option long_options[] = {
  {"help", no_argument, 0, 'h'},
  {"port", required_argument, 0, 'p'},
  {"query", required_argument, 0, 'q'},
  {"verbose", no_argument, 0, 'v'},
  {"very-verbose", no_argument, 0, 'V'},
  {0, 0, 0, 0}};


void printUsage(char* progname) {
  //fprintf(stderr,"Usage: cat <RAW_FILE> | %s [OPTIONS]... [MODEL_FILE1] [MODEL_FILE2] ...\n",progname);
  //fprintf(stderr,"Runs the speech recognition system on RAW_FILE using the models in MODEL_FILE1, 2, etc.\n");
  fprintf(stderr," System variable Settings:\n");
  fprintf(stderr,"  -h, --help\t\tPrint this message\n");
  //fprintf(stderr,"  -p, --port=PORT\tListens for updates on port PORT. (default %d.)\n",UPDATE_PORT_NUM);
  //fprintf(stderr,"  -q, --quer=PORT\tListens for queries on port PORT. (default %d.)\n",QUERY_PORT_NUM);
  fprintf(stderr,"  -v, --verbose\t\tTurns on NOISY output. (default off.)\n");
  fprintf(stderr,"  -V, --very-verbose\tTurns on VERY_NOISY and NOISY output. (default off.)\n");
}


int main (int nArgs, char* argv[]) {

//  int UPDATE_PORT_NUM = 0;
//  int QUERY_PORT_NUM  = 0;

  PgModel mPg;
  PwModel mPw;
  PModel  mP;
  WModel  mW;
  GModel  mG;
  GModel  mGr;
  GGModel mGG;
  CCModel mCC;
  CuModel mCu;
  CuModel mCn;

  const int K_MAX=20;

  ////////// PART 0: COMMAND-LINE OPTIONS

  // Parse command-line options
  char* progname = strdup(argv[0]);
  int option_index = 0;
  char opt;
  while( (opt = getopt_long(nArgs, argv, "hp:q:Vv", long_options, &option_index)) != -1 ) {
    switch(opt) {
    case 'h': printUsage(progname);
              exit(0);
    //case 'p': UPDATE_PORT_NUM  = atoi(optarg); break;
    //case 'q': QUERY_PORT_NUM = atoi(optarg); break;
    //case 'V': WorldModel::OUTPUT_VERY_VERBOSE = true; break; // do break.
    //case 'v': WorldModel::OUTPUT_VERBOSE = true; break;
    default: break;
    }
  }
  // Complain if too few args...
  if (optind > nArgs ) {
    printUsage(progname);
    exit(0);
  }


  ////////// PART I. LOAD MODEL (FROM STDIN)

  FILE* pf = stdin;

  int c=' '; int i=0; int line=1; String sBuff(1000);                          // Lookahead/ctrs/buffers
  CONSUME_ALL ( pf, c, WHITESPACE(c), line);                                   // Get to first record
  while ( c!=-1 && c!='\0' && c!='\5' ) {                                      // For each record
    CONSUME_STR ( pf, c, (c!='\n' && c!='\0' && c!='\5'), sBuff, i, line );    //   Consume line
    StringInput si(sBuff.c_array());
    if ( !( sBuff[0]=='#' ||                                                   //   Accept comments/fields
	    (si>>"Pg ">>mPg>>"\0"!=NULL) ||
	    (si>>"Pw ">>mPw>>"\0"!=NULL) ||
	    (si>>"P " >>mP >>"\0"!=NULL) ||
	    (si>>"W " >>mW >>"\0"!=NULL) ||
            (si>>"GG ">>mGG>>"\0"!=NULL) ||
	    (si>>"Gr ">>mGr>>"\0"!=NULL) ||
	    (si>>"G " >>mG >>"\0"!=NULL) ||
	    (si>>"CC ">>mCC>>"\0"!=NULL) ||
	    (si>>"Cu ">>mCC>>"\0"!=NULL) ) )
      cerr<<"\nERROR: unable to process line "<<line<<": "<<sBuff<<"\n";
    CONSUME_ALL ( pf, c, WHITESPACE(c), line);                                 //   Consume whitespace
  }

  //mGG.dump(cout,"GG");
  //G gI; for ( bool bI=gI.setFirst(); bI; bI=gI.setNext() )
  //        cerr<<"!!! "<<gI<<"\n";
  //exit(0);


  ////////// PART II. MODEL TRANSFORMS

  //// 1. Define PCFG right star probs...

  typedef CPT3DModel<G,G,K,Prob> GdModel;
  GdModel mGrstar;
  typedef CPT2DModel<G,G,Prob> GModel;
  GModel mGrstarsum;
  // For chains of increasing length k...
  for ( int k=0; k<=K_MAX; k++ ) {
    cerr<<"1,"<<k<<"...\n";
    // For each A_0...
    for ( GGModel::const_iterator iGG=mGG.begin(); iGG!=mGG.end(); iGG++ ) {
      G gA0 = iGG->first.getX1();

      // Base case...
      if ( 0==k ) {
        mGrstar.setProb ( gA0, gA0, k ) = 1.0;
      }
      // Induction step...
      else {
        // For each A_k-1, A_k...
        GdModel::IterVal gAkdec; for ( bool bAkdec=mGrstar.setFirst(gAkdec,gA0,k-1); bAkdec; bAkdec=mGrstar.setNext(gAkdec,gA0,k-1) ) {
	  GGModel::IterVal ggAk; for ( bool bAk=mGG.setFirst(ggAk,gAkdec); bAk; bAk=mGG.setNext(ggAk,gAkdec) ) {
	    if ( ggAk.second != G_NONE ) {
	      mGrstar.setProb ( ggAk.second, gA0, k ) += mGrstar.getProb(gAkdec,gA0,k-1) * mGG.getProb(ggAk,gAkdec);
	    }
	  }
        }
      }

      // If k sufficiently large...
      if ( k>=2 ) {
	// For each output, calc running total...
	GModel::IterVal gAk; for ( bool bAk=mGrstar.setFirst(gAk,gA0,k); bAk; bAk=mGrstar.setNext(gAk,gA0,k) ) {
	  mGrstarsum.setProb ( gAk, gA0 ) += mGrstar.getProb(gAk,gA0,k);
	}
      }
    }
  }

  //mGrstarsum.dump(cout,"G*");
  //exit(0);


  ////////// 2. DEFINE RIGHT-CORNER GRAMMAR PROBS...

  SimpleHash<G,B> hIsLeft;
  G gA; for ( bool bA=gA.setFirst(); bA; bA=gA.setNext() ) {
    GGModel::IterVal ggBC; for ( bool bBC=mGG.setFirst(ggBC,gA); bBC; bBC=mGG.setNext(ggBC,gA) ) {
      hIsLeft.set(ggBC.first)=B_1;
    }
  }

  G gB;

  //// 2a. Define RC-PCFG probs (at root, root is preterminal)...

  // For each B...
  for ( bool bB=gB.setFirst(); bB; bB=gB.setNext() ) if ( B_1==hIsLeft.get(gB) ) {
    cerr<<"2a,"<<gB<<"...\n";
    String sB; sB<<gB; C cB(sB.c_array());
    if ( mGG.getProb(GG_NONE,gB) > Prob() ) {
      mCC.setProb ( CC_NONE, cB ) += mG.getProb(gB) * mGG.getProb(GG_NONE,gB);
//         cout<<" "<<cB<<" -a> "<<CC_NONE
//             <<"("<<                   (mG.getProb(gP) * mGG.getProb(ggBQ,gP) * mGG.getProb(GG_NONE,gB))<<") = "
//             <<gP<<"("<<mG.getProb(gP)<<") -> "
//             <<ggBQ<<"("<<mGG.getProb(ggBQ,gP)<<") L> "
//             <<GG_NONE<<"("<<mGG.getProb(GG_NONE,gB)<<")\n";
    }
  }

  //// 2b. Define RC-PCFG probs (at root, right child is preterminal)...

  // For each B...
  for ( bool bB=gB.setFirst(); bB; bB=gB.setNext() ) if ( B_1==hIsLeft.get(gB) ) {
    cerr<<"2b,"<<gB<<"...\n";
    String sB; sB<<gB; C cB(sB.c_array());
    // For each U,C by rule from B...
    GGModel::IterVal ggUC; for ( bool bUC=mGG.setFirst(ggUC,gB); bUC; bUC=mGG.setNext(ggUC,gB) ) if ( ggUC.second != G_NONE ) {
      G& gU = ggUC.first;
      G& gC = ggUC.second;
      if ( mGG.getProb(GG_NONE,gC) > Prob() ) {
	String sU;    sU<<gU;                               C cU(sU.c_array());
	String sC;    sC<<gC;                               C cC(sC.c_array());
	////String sBC;   sBC<<gB<<"/"<<gC;                     C cBC(sBC.c_array());
	String sUBCC; sUBCC<<gB<<"/"<<gC; C cUBCC(sUBCC.c_array());
	mCC.setProb ( CC(cUBCC,cC),  cB    ) += mG.getProb(gB) * mGG.getProb(ggUC,gB) * mGG.getProb(GG_NONE,gC);
	mCC.setProb ( CC(cU,C_NONE), cUBCC ) += mG.getProb(gB) * mGG.getProb(ggUC,gB) * mGG.getProb(GG_NONE,gC);
	mCC.setProb ( CC_NONE,       cC    ) += mG.getProb(gB) * mGG.getProb(ggUC,gB) * mGG.getProb(GG_NONE,gC);
	mCu.setProb ( cU, cUBCC ) = 1.0;
	mCn.setProb ( cU, cUBCC ) = 1.0;
      }
    }
  }

  //// 2c. Define RC-PCFG probs (at root, right grandchild is preterminal)...

  // For each B...
  for ( bool bB=gB.setFirst(); bB; bB=gB.setNext() ) if ( B_1==hIsLeft.get(gB) ) {
    cerr<<"2c,"<<gB<<"...\n";
    String sB; sB<<gB; C cB(sB.c_array());
    // For each U,C by rule from B...
    GGModel::IterVal ggUC; for ( bool bUC=mGG.setFirst(ggUC,gB); bUC; bUC=mGG.setNext(ggUC,gB) ) if ( ggUC.second != G_NONE ) {
      G& gU = ggUC.first;
      G& gC = ggUC.second;
      String sU;  sU<<gU;           C cU(sU.c_array());
      //String sBC; sBC<<gB<<"/"<<gC; C cBC(sBC.c_array());
      // For each V,D by rule from C...
      GGModel::IterVal ggVD; for ( bool bVD=mGG.setFirst(ggVD,gC); bVD; bVD=mGG.setNext(ggVD,gC) ) if ( ggVD.second != G_NONE ) {
	G& gV = ggVD.first;
	G& gD = ggVD.second;
	if ( mGG.getProb(GG_NONE,gD) > Prob() ) {
	  String sV;    sV<<gV;             C cV(sV.c_array());
	  String sD;    sD<<gD;             C cD(sD.c_array());
	  String sBD;   sBD<<gB<<"/"<<gD;   C cBD(sBD.c_array());
	  String sBDD;  sBDD<<gB<<"/"<<gD;  C cBDD(sBDD.c_array());
	  String sUBCV; sUBCV<<gB<<"/"<<gC; C cUBCV(sUBCV.c_array());
	  mCC.setProb ( CC(cBDD,cD),  cB   )   += mG.getProb(gB) * mGG.getProb(ggUC,gB) * mGG.getProb(ggVD,gC) * mGG.getProb(GG_NONE,gD);
	  mCC.setProb ( CC(cUBCV,cV), cBDD )   += mG.getProb(gB) * mGG.getProb(ggUC,gB) * mGG.getProb(ggVD,gC) * mGG.getProb(GG_NONE,gD);
	  mCC.setProb ( CC(cU,C_NONE), cUBCV ) += mG.getProb(gB) * mGG.getProb(ggUC,gB) * mGG.getProb(ggVD,gC) * mGG.getProb(GG_NONE,gD);
	  mCC.setProb ( CC_NONE,      cD   )   += mG.getProb(gB) * mGG.getProb(ggUC,gB) * mGG.getProb(ggVD,gC) * mGG.getProb(GG_NONE,gD);
	  mCu.setProb ( cU,  cUBCV ) = 1.0;
	  mCn.setProb ( cU,  cUBCV ) = 1.0;
	  mCn.setProb ( cBD, cBDD  ) = 1.0;
//             cout<<" "<<cB<<" -c> "<<CC(cBDD,cD)
//                 <<"("<<                          (mG.getProb(gP) * mGG.getProb(ggBQ,gP) * mGG.getProb(ggUC,gB) * mGG.getProb(ggVD,gC) * mGG.getProb(GG_NONE,gD))<<") = "
//                 <<gP<<"("<<mG.getProb(gP)<<") -> "
//                 <<ggBQ<<"("<<mGG.getProb(ggBQ,gP)<<") L> "
//                 <<ggUC<<"("<<mGG.getProb(ggUC,gB)<<") R> "
//                 <<ggVD<<"("<<mGG.getProb(ggVD,gC)<<") R> "
//                 <<GG_NONE<<"("<<mGG.getProb(GG_NONE,gD)<<")\n";
//             cout<<" "<<cD<<" -c> "<<CC_NONE
//                 <<"("<<                          (mG.getProb(gP) * mGG.getProb(ggBQ,gP) * mGG.getProb(ggUC,gB) * mGG.getProb(ggVD,gC) * mGG.getProb(GG_NONE,gD))<<") = "
//                 <<gP<<"("<<mG.getProb(gP)<<") -> "
//                 <<ggBQ<<"("<<mGG.getProb(ggBQ,gP)<<") L> "
//                 <<ggUC<<"("<<mGG.getProb(ggUC,gB)<<") R> "
//                 <<ggVD<<"("<<mGG.getProb(ggVD,gC)<<") R> "
//                 <<GG_NONE<<"("<<mGG.getProb(GG_NONE,gD)<<")\n";
	}
      }
    }
  }

  //// 2d. Define RC-PCFG probs (at root, neither right child/grandchild is preterminal)...

  // For each B...
  for ( bool bB=gB.setFirst(); bB; bB=gB.setNext() ) if ( B_1==hIsLeft.get(gB) ) {
    cerr<<"2d,"<<gB<<"...\n";
    String sB; sB<<gB; C cB(sB.c_array());
    // For each U,C by rule from B...
    GGModel::IterVal ggUC; for ( bool bUC=mGG.setFirst(ggUC,gB); bUC; bUC=mGG.setNext(ggUC,gB) ) if ( ggUC.second != G_NONE ) {
      G& gU = ggUC.first;
      G& gC = ggUC.second;
      String sU;  sU<<gU;           C cU(sU.c_array());
      //String sBC; sBC<<gB<<"/"<<gC; C cBC(sBC.c_array());
      // For each V,D by rule from C...
      GGModel::IterVal ggVD; for ( bool bVD=mGG.setFirst(ggVD,gC); bVD; bVD=mGG.setNext(ggVD,gC) ) if ( ggVD.second != G_NONE ) {
	G& gV = ggVD.first;
	G& gD = ggVD.second;
	String sV;  sV<<gV;           C cV(sV.c_array());
	String sD;  sD<<gD;           C cD(sD.c_array());
	String sBD; sBD<<gB<<"/"<<gD; C cBD(sBD.c_array());
//	// For each W,E by rule from D...
//	GGModel::IterVal ggWE; for ( bool bWE=mGG.setFirst(ggWE,gD); bWE; bWE=mGG.setNext(ggWE,gD) ) if ( ggWE.second != G_NONE ) {
//	  G& gW = ggWE.first;
//	  String sW;    sW<<gW;             C cW(sW.c_array());
	  String sUBCV; sUBCV<<gB<<"/"<<gC; C cUBCV(sUBCV.c_array());
	  String sBDW;  sBDW<<gB<<"/"<<gD;  C cBDW(sBDW.c_array());
	  mCC.setProb ( CC(cUBCV,cV), cBDW )   += mG.getProb(gB) * mGG.getProb(ggUC,gB) * mGG.getProb(ggVD,gC); //* mGG.getProb(ggWE,gD);
	  mCC.setProb ( CC(cU,C_NONE), cUBCV ) += mG.getProb(gB) * mGG.getProb(ggUC,gB) * mGG.getProb(ggVD,gC); //* mGG.getProb(ggWE,gD);
	  mCu.setProb ( cU,  cUBCV ) = 1.0;
	  mCn.setProb ( cU,  cUBCV ) = 1.0;
	  mCn.setProb ( cBD, cBDW  ) = 1.0;
//	}
      }
    }
  }

  //// 2e. Define RC-PCFG probs (not at root, right child preterminal)...

  // For each B...
  for ( bool bB=gB.setFirst(); bB; bB=gB.setNext() ) if ( B_1==hIsLeft.get(gB) ) {
    cerr<<"2e,"<<gB<<"...\n";
    String sB; sB<<gB; C cB(sB.c_array());
    // For each E by right-star from B...
    GdModel::IterVal gE; for ( bool bE=mGrstarsum.setFirst(gE,gB); bE; bE=mGrstarsum.setNext(gE,gB) ) {
      String sE;  sE<<gE;           C cE(sE.c_array());
      String sBE; sBE<<gB<<"/"<<gE; C cBE(sBE.c_array());
      // For each X,F by rule from E...
      GGModel::IterVal ggXF; for ( bool bXF=mGG.setFirst(ggXF,gE); bXF; bXF=mGG.setNext(ggXF,gE) ) if ( ggXF.second != G_NONE ) {
	G& gX = ggXF.first;
	G& gF = ggXF.second;
	if ( mGG.getProb(GG_NONE,gF) > Prob() ) {
	  String sX;   sX<<gX;            C cX(sX.c_array());
	  String sF;   sF<<gF;            C cF(sF.c_array());
	  String sBF;  sBF<<gB<<"/"<<gF;  C cBF(sBF.c_array());
	  String sBFF; sBFF<<gB<<"/"<<gF; C cBFF(sBFF.c_array());
	  String sBEX; sBEX<<gB<<"/"<<gE; C cBEX(sBEX.c_array());
	  mCC.setProb ( CC(cBFF,cF), cB   ) += mG.getProb(gB) * mGrstarsum.getProb(gE,gB) * mGG.getProb(ggXF,gE) * mGG.getProb(GG_NONE,gF);
	  mCC.setProb ( CC(cBEX,cX), cBFF ) += mG.getProb(gB) * mGrstarsum.getProb(gE,gB) * mGG.getProb(ggXF,gE) * mGG.getProb(GG_NONE,gF);
	  mCC.setProb ( CC_NONE,     cF   ) += mG.getProb(gB) * mGrstarsum.getProb(gE,gB) * mGG.getProb(ggXF,gE) * mGG.getProb(GG_NONE,gF);
	  mCn.setProb ( cBE, cBEX ) = 1.0;
	  mCn.setProb ( cBF, cBFF  ) = 1.0;
//             cout<<" "<<cB<<" -e> "<<CC(cBFF,cF)
//                 <<"("<<                         (mG.getProb(gP) * mGG.getProb(ggBQ,gP) * mGrstarsum.getProb(gE,gB) * mGG.getProb(ggXF,gE) * mGG.getProb(GG_NONE,gF))<<") = "
//                 <<gP<<"("<<mG.getProb(gP)<<") -> "
//                 <<ggBQ<<"("<<mGG.getProb(ggBQ,gP)<<") L..> "
//                 <<gE<<"("<<mGrstarsum.getProb(gE,gB)<<") R> "
//                 <<ggXF<<"("<<mGG.getProb(ggXF,gE)<<") R> "
//                 <<GG_NONE<<"("<<mGG.getProb(GG_NONE,gF)<<")\n";
//             cout<<" "<<cF<<" -e> "<<CC_NONE
//                 <<"("<<                         (mG.getProb(gP) * mGG.getProb(ggBQ,gP) * mGrstarsum.getProb(gE,gB) * mGG.getProb(ggXF,gE) * mGG.getProb(GG_NONE,gF))<<") = "
//                 <<gP<<"("<<mG.getProb(gP)<<") -> "
//                 <<ggBQ<<"("<<mGG.getProb(ggBQ,gP)<<") L..> "
//                 <<gE<<"("<<mGrstarsum.getProb(gE,gB)<<") R> "
//                 <<ggXF<<"("<<mGG.getProb(ggXF,gE)<<") R> "
//                 <<GG_NONE<<"("<<mGG.getProb(GG_NONE,gF)<<")\n";
	}
      }
    }
  }

  //// 2f. Define RC-PCFG probs (not at root, right child not preterminal)...

  // For each B...
  for ( bool bB=gB.setFirst(); bB; bB=gB.setNext() ) if ( B_1==hIsLeft.get(gB) ) {
    cerr<<"2f,"<<gB<<"...\n";
    String sB; sB<<gB; C cB(sB.c_array());
    // For each E by right-star from B...
    GdModel::IterVal gE; for ( bool bE=mGrstarsum.setFirst(gE,gB); bE; bE=mGrstarsum.setNext(gE,gB) ) {
      String sE;  sE<<gE;           C cE(sE.c_array());
      String sBE; sBE<<gB<<"/"<<gE; C cBE(sBE.c_array());
      // For each X,F by rule from E...
      GGModel::IterVal ggXF; for ( bool bXF=mGG.setFirst(ggXF,gE); bXF; bXF=mGG.setNext(ggXF,gE) ) if ( ggXF.second != G_NONE ) {
	G& gX = ggXF.first;
	G& gF = ggXF.second;
	String sX;  sX<<gX;           C cX(sX.c_array());
	String sBF; sBF<<gB<<"/"<<gF; C cBF(sBF.c_array());
//	// For each Y,G by rule from F...
//	GGModel::IterVal ggYG; for ( bool bYG=mGG.setFirst(ggYG,gF); bYG; bYG=mGG.setNext(ggYG,gF) ) if ( ggYG.second != G_NONE ) {
//	  G& gY = ggYG.first;
	  String sBFY; sBFY<<gB<<"/"<<gF; C cBFY(sBFY.c_array());
	  String sBEX; sBEX<<gB<<"/"<<gE; C cBEX(sBEX.c_array());
	  mCC.setProb ( CC(cBEX,cX), cBFY ) += mG.getProb(gB) * mGrstarsum.getProb(gE,gB) * mGG.getProb(ggXF,gE); // * mGG.getProb(ggYG,gF);
	  mCn.setProb ( cBE, cBEX ) = 1.0;
	  mCn.setProb ( cBF, cBFY ) = 1.0;
//	}
      }
    }
  }

  mCC.normalize();

  mG.dump (cout,"G" );
  mGr.dump(cout,"Gr");
  mCu.dump(cout,"Cu");
  mCn.dump(cout,"Cn");
  mCC.dump(cout,"CC");

  mPg.dump(cout,"Pg");
  mPw.dump(cout,"Pw");
  mP.dump (cout,"P" );
  mW.dump (cout,"W" );
}
