from __future__ import annotations

import os
import shutil
from abc import ABC, abstractmethod
from functools import lru_cache
from pathlib import Path
from typing import List, Optional, Tuple, Union

import numpy as np
import pandas as pd

from hedal.block import Block
from hedal.context import Context
from hedal.core import config
from hedal.core.config import PackingType


class Object(ABC):
    def __init__(
        self,
        context: Context,
        path: Optional[Path] = None,
        encrypted: bool = False,
        type: PackingType = PackingType.FRAME,
    ):
        """Object class: abstract class which can be regarded as a sequence of blocks.

        Args:
            context (Context): Context object.
            path (Path, optional): Path to save object. Defaults to config.temp_path().
            encrypted (bool, optional): Whether encrypted. Defaults to False.
            type (PackingType, optional): Packing type of the object. Defaults to PackingType.FRAME.
        """
        self.context: Context = context
        if path is None:
            self._path: Path = config.temp_path()
        else:
            self._path: Path = Path(path)
        self.encrypted: bool = encrypted
        self.type: PackingType = type

        self.block_list: List[Block] = []

    @property
    def num_slots(self) -> int:
        return self.context.num_slots

    @property
    def block_shape(self) -> Tuple[int, int]:
        if self.type == PackingType.FRAME:
            return (self.context.num_slots, 1)
        elif self.type == PackingType.MATRIX:
            return self.context.shape
        else:
            raise TypeError("Invalid type")

    @property
    def path(self) -> Path:
        return Path(self._path)

    @path.setter
    def path(self, new_path: Path) -> None:
        self._path = Path(new_path)

    @property
    def parent_path(self) -> Path:
        return config.parent_path(self)

    @property
    def name(self) -> str:
        return config.name(self)

    @name.setter
    def name(self, new_name: str) -> None:
        self.rename(new_name)

    @property
    def level(self) -> int:
        return self.block_list[0].level

    def set_block_list(self, block_list: List[Block]) -> None:
        self.block_list = block_list

    def copy_memory(self) -> Object:
        raise NotImplementedError

    def level_down(self, target_level: int, inplace: bool = True) -> Object:
        if not inplace:
            obj = self.copy_memory()
        else:
            obj = self
        for block in obj:
            block.level_down(target_level)
        return obj

    def block_path(self, idx: int) -> str:
        return config.block_path(self, idx)

    @lru_cache(maxsize=4)
    def block(self, idx: int) -> Block:
        return self.block_list[idx]

    def save(self, dst_parent_path: Optional[Union[str, Path]] = None) -> None:
        """Save object to the destination parent path.

        Args:
            dst_parent_path (Optional[Union[str, Path]], optional): Destination parent path. Defaults to None.
        """
        if dst_parent_path is None:
            dst_parent_path = self.parent_path
        if isinstance(dst_parent_path, str):
            dst_parent_path = Path(dst_parent_path)
        dst_path = dst_parent_path / self.name
        if not dst_path.exists():
            os.makedirs(dst_path, mode=0o775, exist_ok=True)
        for idx, block in enumerate(self):
            block_path = dst_path / os.path.basename(self.block_path(idx))
            block.save(block_path)

    def move(self, dst_parent_path: Path) -> Object:
        dst_parent_path = Path(dst_parent_path)
        if not self.parent_path == dst_parent_path:
            dst_path = dst_parent_path / self.name
            if dst_path.exists():
                raise OSError("Already exists:", dst_path)
            shutil.move(self.path, dst_path)
            self.path = dst_path
        return self

    def rename(self, new_name: str) -> Object:
        if not self.name == new_name:
            new_path = self.parent_path / new_name
            if new_path.exists():
                raise OSError("Already exists:", new_path)
            self.path.rename(new_path)
            self.path = new_path
        return self

    def remove(self) -> None:
        shutil.rmtree(self.path)

    def encrypt(self) -> None:
        if self.encrypted:
            raise Exception("Already encrypted")
        for block in self:
            block.encrypt()
        self.encrypted = True

    def decrypt(self) -> None:
        if not self.encrypted:
            raise Exception("Already decrypted")
        for block in self:
            block.decrypt()
        self.encrypted = False

    def bootstrap(self, one_slot: bool = False, complex: bool = False) -> None:
        if complex or one_slot:
            for block in self:
                block.bootstrap(one_slot=one_slot, complex=complex)
        else:
            # bootstrap two blocks at once
            num_blocks = len(self.block_list)
            for i in range(0, num_blocks // 2):
                Block.bootstrap_two_ctxts(self.block(2 * i), self.block(2 * i + 1))
            if num_blocks % 2 == 1:
                self[num_blocks - 1].bootstrap()

    def need_bootstrap(self, cost_per_iter: int = 2) -> bool:
        if self.encrypted:
            return self.level - cost_per_iter < self.context.min_level_for_bootstrap
        else:
            return False

    def __delete__(self, instance) -> None:
        shutil.rmtree(self.path)

    def __getitem__(self, idx: int) -> Block:
        return self.block(idx)

    def __setitem__(self, idx: int, block: Block):
        self.block_list[idx] = block

    def __iter__(self) -> Object:
        self._idx = 0
        return self

    def __next__(self) -> Block:
        if self._idx >= len(self):
            raise StopIteration
        idx = self._idx
        self._idx += 1
        return self.block(idx)

    @staticmethod
    @abstractmethod
    def from_path(context: Context, path: Path, **kwargs) -> Object:
        pass

    @abstractmethod
    def copy(self, dst_path: Optional[Path] = None) -> Object:
        pass

    @abstractmethod
    def __len__(self) -> int:
        pass

    @staticmethod
    def from_ndarray(context: Context, array: np.ndarray, path: Optional[Path] = None):
        raise NotImplementedError

    def to_ndarray(self) -> np.ndarray:
        raise NotImplementedError

    @staticmethod
    def from_series(context: Context, series: pd.Series, path: Optional[Path] = None):
        raise NotImplementedError

    def to_series(self) -> pd.Series:
        raise NotImplementedError
