import subprocess
import time
import gc
import os
import sys
import time
import argparse
import numpy as np
import json
import signal

from py4j.protocol import Py4JNetworkError, Py4JError

from py4j.java_gateway import JavaGateway, GatewayParameters
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy import Column, Integer, String, Boolean, Numeric, Table, PrimaryKeyConstraint
import sqlalchemy
import pandas as pd

from GapSelectionStrategy import GapSelectionStrategy
from GapSizeStrategy import GapSizeStrategy
from StandardStrategy import StandardStrategy
from StrategyGreedy import GreedyStrategy
from nltk.corpus import gutenberg,reuters,brown
import nltk
import analyse as an


def evalClassification(prediction,target):
    '''
    This method implements the desired proficiency metric and maps the prediction and the target on to it.
    If the test is to easy : return (0,-1]
    If the test is to hard : return [0,1)
    If the test fits the target level: return 0
    Following Metric is suggested by \citet[p.136]{Beinborn2016}
        1: Very easy gap (error rate ≤ 0.25)
        2: Easy gap (0.25 < error rate ≤ 0.5)
        3: Medium gap (0.5 < error rate ≤ 0.75)
        4: Difficult gap (error rate > 0.75)
    :param prediction: prediction outcome of the generated c-test
    :param target: desired proficiency level of the test
    :return: [-1,1] difference between average of the prediction and target (positive if to high, negative if to low)
    '''
    avg = sum(prediction) / len(prediction)

    if target == "very easy":
        if avg <0.25:
            return (0,[p-0.15 for p in prediction])
        else: return (avg-0.15,[p-0.15 for p in prediction])
    elif target == "easy":
        if 0.25<=avg <0.5:
            return (0,[p-0.325 for p in prediction])
        else: return (avg-0.325,[p-0.325 for p in prediction])
    elif target == "medium":
        if 0.5<=avg <0.75:
            return (0,[p-0.625 for p in prediction])
        else: return (avg-0.625,[p-0.625 for p in prediction])
    elif target == "difficult":
        if 0.75<=avg <1:
            return (0,[p-0.875 for p in prediction])
        else: return (avg-0.875,[p-0.875 for p in prediction])
    else:
        print("The target parameter is not available. Possibile parameter: very easy,easy,medium,difficult, your input: ", target)


def evalCEFR(prediction,target):
    '''
    This method implements the desired proficiency metric and maps the prediction and the target on to it.
    If the test is to easy : return (0,-1]
    If the test is to hard : return [0,1)
    If the test fits the target level: return 0
    Following Metric is suggested by \citet[p.136]{Beinborn2016}
        1: Very easy gap (error rate ≤ 0.25)
        2: Easy gap (0.25 < error rate ≤ 0.5)
        3: Medium gap (0.5 < error rate ≤ 0.75)
        4: Difficult gap (error rate > 0.75)
    :param prediction: prediction outcome of the generated c-test
    :param target: desired proficiency level of the test
    :return: [-1,1]
    '''
    if target == "A":
        if sum(prediction) / len(prediction) <0.2:
            return 0
        else: return 1
    elif target == "B1":
        if sum(prediction) / len(prediction) <0.5:
            return 0
        else: return 1
    elif target == "B2":
        if sum(prediction) / len(prediction) <0.65:
            return 0
        else: return 1
    elif target == "C1":
        if sum(prediction) / len(prediction) <85:
            return 0
        else: return 1
    elif target == "C2":
        if sum(prediction) / len(prediction) <1:
            return 0
        else: return 1
    else:
        print("The target parameter is not available. Possibile parameter: A,B1,B2,C1,C2 your input: ",target)
        return 1

def evalMeanVariance(prediction,mean_std):
    '''
    This method implements the desired proficiency metric and maps the prediction and the target on to it.
    If the test is to easy : return (0,-1]
    If the test is to hard : return [0,1)
    If the test fits the target level: return 0
    Following Metric is suggested by \citet[p.136]{Beinborn2016}
        1: Very easy gap (error rate ≤ 0.25)
        2: Easy gap (0.25 < error rate ≤ 0.5)
        3: Medium gap (0.5 < error rate ≤ 0.75)
        4: Difficult gap (error rate > 0.75)
    :param prediction: prediction outcome of the generated c-test
    :param target: desired proficiency level of the test
    :return: [-1,1]
    '''
    try:
        avg = sum(prediction) / len(prediction)
        print('Avg: ',avg)
        if avg <= mean_std[0] +mean_std[1] and avg >= mean_std[0] -mean_std[1]:
            return (0,0)
        else:
            return (avg-mean_std[0],[p-mean_std[0] for p in prediction])

    except Exception as e:
        print(e)

def load_data(corpus_name):
    result = []
    file_ids = []
    if corpus_name == 'gutenberg':
        file_ids = gutenberg.fileids()
        result = [' '.join(gutenberg.words(fileids=[f])) for f in sorted(file_ids)]
    elif corpus_name == 'brown':
        file_ids = brown.fileids()
        result = [' '.join(brown.words(fileids=[f])) for f in sorted(file_ids)]
    else:
        file_ids = reuters.fileids()
        result = [reuters.raw(fileids=[f]) for f in sorted(file_ids)]
    return result, file_ids


def main():
    parser = argparse.ArgumentParser(description='Ctest Generator for different strategies')
    parser.add_argument('--strategy', default='SEL', help='Strategy to create C-test. DEF, SIZE, or SEL')
    parser.add_argument('--target', type=float, default=0.4, help='Target error rate for the generated C-test.')
    parser.add_argument('--infile', default='data/cd10_2.txt', help='Input file with the text bases')
    parser.add_argument('--ctest', default='results/', help='Path to store the resulting ctests')
    args = parser.parse_args()

    try:
        os.environ['DKPRO_HOME']
    except KeyError:
        print("DKPRO_HOME enviroment variable is not set!")
        print("Finishing ...")
        sys.exit()

    proc = subprocess.Popen('java -jar DifficultyPrediction.jar', shell=True,stdin=None, stdout=subprocess.PIPE, stderr=None, close_fds=True, preexec_fn=os.setsid) 
    gateway = JavaGateway(gateway_parameters=GatewayParameters(enable_memory_management=True),start_callback_server=True)
    gateway.entry_point
    time.sleep(2)
    log = open(args.infile,'r')
    text = log.read().strip()
    log.close()

    try: 
        generated_ctest = ''

        if args.strategy == 'SIZE':
            gsize = GapSizeStrategy([text], evalMeanVariance, (args.target, 0.0) , gateway.entry_point)
            generated_ctest = gsize.retrieve_ctest()
        elif args.strategy == 'SEL':
            gsel = GapSelectionStrategy([text], evalMeanVariance, (args.target,0.0) , gateway.entry_point)
            generated_ctest = gsel.retrieve_ctest()
        else:
            # DEF is the default strategy. 
            # DEF has no target C-test difficulty.
            gdef = StandardStrategy([text], evalMeanVariance, "egal", gateway.entry_point)
            generated_ctest = gdef.retrieve_ctest()

        if not generated_ctest:
            print("The provided input file is to short to generate a C-test.")
            gateway.shutdown()
            gc.collect()
            proc.terminate()
            os.killpg(os.getpgid(proc.pid), signal.SIGTERM)
            sys.exit()

        # include the estimated difficulties and write the data:
        outlog = open(args.ctest + args.infile.split('/')[-1].replace('.txt','_{}.txt'.format(args.strategy)),'w')
        index = 0
        for line in generated_ctest[0].split('\n'):
            if '0.0' in line:
                outlog.write(line.replace('0.0','{}'.format(round(generated_ctest[-1][index],4)))+'\n')
                index += 1
            else:
                outlog.write(line+'\n')
        outlog.write("===== difficulty: {} =====".format(np.mean(generated_ctest[-1])))
        outlog.close()

        gc.collect()

    except (Py4JNetworkError,Py4JError,ConnectionRefusedError):
        gateway.shutdown()
        gc.collect()
        proc.terminate()
        os.killpg(os.getpgid(proc.pid), signal.SIGTERM)
        sys.exit()

    print('=========================================== DONE ===========================================')

    gc.collect()
    gateway.shutdown()
    proc.terminate()
    os.killpg(os.getpgid(proc.pid), signal.SIGTERM)
    sys.exit()


if __name__ == '__main__':
    main()
