import carla
import numpy as np



def _numpy(carla_vector, normalize=False):
    """
    convert a carla vector to a numpy array
    """
    result = np.float32([carla_vector.x, carla_vector.y])
    if normalize:
        return result / (np.linalg.norm(result) + 1e-4)
    return result


def _orientation(yaw):
    """
    get the orientation of a given yaw
    """
    return np.float32([np.cos(np.radians(yaw)), np.sin(np.radians(yaw))])


def get_angle_to(pos, theta, target):
    """
    get angle to a given target from a given position, given the current orientation
    """
    R = np.array([[np.cos(theta), -np.sin(theta)], [np.sin(theta), np.cos(theta)]])
    aim = R.T.dot(target - pos)
    angle = -np.degrees(np.arctan2(-aim[1], aim[0]))
    angle = 0.0 if np.isnan(angle) else angle
    return angle


def get_collision(p1, v1, p2, v2):
    """
    determine if and where two moving objects will collide based on their current positions and movement directions
    :param p1: The position vector of the first object.
    :param v1: The direction vector of the first object, representing its movement direction and speed.
    :param p2: The position vector of the second object.
    :param v2: The direction vector of the second object, also representing its movement direction and speed.
    :return:
    """
    A = np.stack([v1, -v2], 1)
    b = p2 - p1

    if abs(np.linalg.det(A)) < 1e-3:
        # The two objects are moving in parallel, or close to it.
        return False, None

    x = np.linalg.solve(A, b)
    # The elements of x represent the scalar multiples of v1 and v2 that need to be added to p1 and p2, respectively,
    # to reach the point of intersection.

    # checks that the intersection point lies within the paths defined by the direction vectors v1 and v2
    # starting from p1 and p2, respectively, within the same timeframe.
    collides = all(x >= 0) and all(x <= 1)

    return collides, p1 + x[0] * v1


def is_point_inside_bounding_box(point, bb_center, bb_extent):
    """
    check if a given point is inside a given bounding box
    """
    A = carla.Vector2D(bb_center.x - bb_extent.x, bb_center.y - bb_extent.y)  # bottom left corner
    B = carla.Vector2D(bb_center.x + bb_extent.x, bb_center.y - bb_extent.y)  # bottom right corner
    D = carla.Vector2D(bb_center.x - bb_extent.x, bb_center.y + bb_extent.y)  # top left corner
    M = carla.Vector2D(point.x, point.y)

    AB = B - A
    AD = D - A
    AM = M - A

    # dot products
    am_ab = AM.x * AB.x + AM.y * AB.y
    ab_ab = AB.x * AB.x + AB.y * AB.y
    am_ad = AM.x * AD.x + AM.y * AD.y
    ad_ad = AD.x * AD.x + AD.y * AD.y

    # The function checks whether the point M lies within the bounding box by ensuring that its projections
    # (am_ab and am_ad) onto the edges are greater than 0 (meaning it's beyond point A along the direction of the edges)
    # and less than the squared lengths of the edges (ab_ab and ad_ad, respectively).
    # This ensures that the point is within the extents of the bounding box in both dimensions.
    return 0 <= am_ab <= ab_ab and 0 <= am_ad <= ad_ad





def find_closest_valid_traffic_light(loc, min_dis, carla_map):
    wp = carla_map.get_waypoint(loc)
