from SPARQLWrapper import SPARQLWrapper, JSON
from typing import List, Literal, Optional
from urllib.error import HTTPError
import json
import time


def get_id_from_uri(uri: str) -> str:
    """Return the ID from a Wikidata URI."""
    return uri.split('/')[-1]


# Wikidata query service setting
wikidata_url = "https://query.wikidata.org/sparql"
user_agent = "" # HTTP User-Agent header
sparql = SPARQLWrapper(wikidata_url, agent=user_agent)


def safe_sparql_request(sparql):
    """Make a SPARQL request with error handling."""
    while True:
        try:
            # Requests that make too many calls may result in a 429 error.
            results = sparql.query().convert()
            return results

        except HTTPError as e:
            if e.code == 429:
                print("429 error occurred. Pausing execution for a while...")
                time.sleep(15)  # Sleep for 15 seconds before retrying.
            else:
                # If the error code is not 429, re-raise the exception.
                raise e


# According to the property id, get the label of the relation.
def get_property_label(property_id: str) -> str:
    """Get the label of a property given its ID."""
    queryString = """
    PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
    PREFIX wikibase: <http://wikiba.se/ontology#>
    PREFIX bd: <http://www.bigdata.com/rdf#>

    SELECT ?propertyLabel WHERE {
    wd:%s rdfs:label ?propertyLabel.
    FILTER(LANG(?propertyLabel) = "en")
    }
    """ % property_id

    sparql.setQuery(queryString)

    sparql.setReturnFormat(JSON)
    results = sparql.query().convert()

    return results['results']['bindings'][0]['propertyLabel']['value']


# According to the item id, get the neighbor triples of the entity.
def get_neighbor_triples(item_id: str) -> str:
    """Get the triples of the neighbors of an entity given its ID."""
    queryString = """
    CONSTRUCT {
      ?s ?p ?o .
      ?p rdfs:label ?pLabel .
      ?o rdfs:label ?oLabel .
    } 
    WHERE { 
      BIND(wd:%s AS ?s) 
      ?s ?p ?o .
      ?wdProp wikibase:directClaim ?p .
      
      SERVICE wikibase:label {
        bd:serviceParam wikibase:language "[AUTO_LANGUAGE],en". 
        ?wdProp rdfs:label ?pLabel .
        ?o rdfs:label ?oLabel .
      }

      FILTER(STRSTARTS(STR(?p), STR(wdt:))) .
      FILTER(STRSTARTS(STR(?o), STR(wd:))) .
      FILTER(!STRSTARTS(STR(?p), STR(wdt:P910))) .
      FILTER(!STRSTARTS(STR(?p), STR(wdt:P1423))) .
    } 
    """ % item_id

    sparql.setQuery(queryString)

    sparql.setReturnFormat(JSON)
    label = sparql.query().convert()

    return label


def init_triples_dict(subject, relation, obj) -> dict:
    """Initialize a dictionary to represent a triple."""
    return {
        "subject": subject,
        "relation": relation,
        "target": obj
    }


# Initialize the keys in the new json.
def initialize_json(json_obj: dict) -> dict:
    """Initialize the JSON object for the new triples."""
    return {
        "case_id": json_obj["case_id"],
        "triples": []
    }


# Get all the neighbor triples of an item as outgoing edges.
def get_entity_triplets(entity_id: str, entity_label: str) -> list:
    """Get the triples of an entity and its neighbors."""
    # Initialize a dictionary to store the mapping between entity URIs and their labels.
    entity_id_to_label = {}
    entity_id_to_label[entity_id] = entity_label

    # Get the object entities.
    results = get_neighbor_triples(entity_id)

    # First, process all the items with label information.
    for result in results["results"]["bindings"]:
        type = result['object']['type']
        if not type == "literal":
            continue
        subject = get_id_from_uri(result['subject']['value'])
        entity_id_to_label[subject] = result['object']['value']

    # Then, process all the edges (relations), and convert them to the required JSON format.
    triples = []
    for result in results["results"]["bindings"]:
        type = result['object']['type']
        if not type == "uri":
            continue

        subject = get_id_from_uri(result['subject']['value'])
        predicate = get_id_from_uri(result['predicate']['value'])
        obj = get_id_from_uri(result['object']['value'])

        triple = init_triples_dict(
            entity_id_to_label[subject],
            entity_id_to_label[predicate],
            entity_id_to_label[obj]
        )

        triples.append(triple)

    return triples


def get_triplets_from_dataset(dataset_path):
    """Get the triples from a dataset."""
    # Read the original dataset.
    with open(dataset_path, 'r') as file:
        dataset = json.load(file)

    # Initialize a counter for storing the number of results.
    counter = 0
    relation_id_dict = {}
    new_json = []

    # Iterate over each case.
    for case in dataset:
        print("Processing Case{}".format(case['case_id']))

        if (counter == 10):
            break

        # Initialize a new JSON object.
        new_data = initialize_json(case)

        # Iterate over each requested_rewrite.
        rewrite = case["requested_rewrite"]

        subject_label = rewrite["subject"]

        # Get the ID and label of the target_new.
        target_id = rewrite["target_new"]["id"]
        target_label = rewrite["target_new"]["str"]

        # Get the property label of the relation.
        if (rewrite["relation_id"] in relation_id_dict):
            relation_label = relation_id_dict[rewrite["relation_id"]]
        else:
            relation_label = get_property_label(rewrite["relation_id"])
            relation_id_dict[rewrite["relation_id"]] = relation_label

        # Delete the old relation ID, and add the new relation label.
        # del rewrite["relation_id"]
        # rewrite["relation_label"] = relation_label

        # Add to the new dataset.
        # new_data["requested_rewrite"].append(rewrite)

        # Get all the triples of the target_new.
        origin_triplet = init_triples_dict(
            subject_label, relation_label, target_label)
        target_neighbors = get_entity_triplets(
            target_id, target_label)

        # Add these triples to the new JSON object.
        new_data["triples"].extend(origin_triplet)
        new_data["triples"].extend(target_neighbors)

        counter += 1

        new_json.extend(new_data)

        # Decide the filename based on the counter.
        if counter % 1000 == 0:
            file_name = f'cf_graph_{counter}_to_{counter+999}.json'
            with open(file_name, 'w') as output_file:
                json.dump(new_json, output_file)
            new_json = []

    # If the last batch of data does not reach 1000 cases, save the remaining results to the final file.
    if counter % 1000 != 0:
        file_name = f'cf_graph_{counter - (counter % 1000)}_to_{counter - 1}.json'
        with open(file_name, 'w') as output_file:
            json.dump(new_json, output_file)
        new_json = []


if __name__ == "__main__":
    get_triplets_from_dataset(
        "/datasets/counterfact.json")
