function [X, unique] = sylvester_nochecks(A, B, C)
% Solve Sylvester equation without input checks.
%
% function [X, notunique] = sylvester_nochecks(A, B, C)
%
% Solves the Sylvester equation A*X + X*B = C, where A is an m-by-m matrix,
% B is an n-by-n matrix, and X and C are two m-by-n matrices. The returned
% flag 'unique' is set to false if the solution is not unique.
%
% This is a stripped-down version of Matlab's own sylvester function that
% bypasses any input checks. This is significantly faster for small m and
% n, which is often useful in Manopt.
%
% See also: sylvester lyapunov_symmetric

% This file is part of Manopt: www.manopt.org.
% Original author: Nicolas Boumal, July 19, 2018
% Contributors: This is a modification of Matlab's built-in sylvester.
% Change log: 

    flag = 'real';
    if ~isreal(A) || ~isreal(B) || ~isreal(C)
        flag = 'complex';
    end

    [QA, TA] = schur(A, flag);
    [QB, TB] = schur(B, flag);
    
    % Solve Sylvester Equation TA*Y + Y*TB = QA'*C*QB.
    [Y, info] = builtin('_sylvester_tri', TA, TB, QA'*C*QB);
    
    unique = (info ~= 1);

    X = QA*Y*QB';
    
end
