#ifndef RULE_H
#define RULE_H

#include <vector>
#include <string>
#include <map>
#include <iostream>

#include "include/keyref.h"

class RuleSet {
    friend std::istream& operator>>(std::istream& is, RuleSet& rs);
    friend std::ostream& operator<<(std::ostream& os, const RuleSet& rs);

public:
    struct Tail;
    using Label = int;
    using Literal = int;
    using State = int;
    using States = std::vector< State >;
    using Equation = std::vector< int >;
    using Equations = std::vector< Equation >;
    using Tails = std::vector< const Tail* >;

    struct Head {
        Label label_index;
        States in_states;
        int out_state_count;

        bool operator<(const Head& rs) const;
    };

    struct Tail {
        int rule_index;
        States out_states;
        Equations equations;
    };

    struct Rule {
        Head head;
        Tail tail;
    };

private:
    std::vector< Rule > rules_list_;
    std::map< KeyRef< const Head >, std::vector< const Tail* > > rules_map_;

    static Tails empty_;

public:
    const Tails& find(const Head& rule_head) const {
        auto iter = rules_map_.find(rule_head);
        if (iter != rules_map_.end())
            return iter->second;
        return empty_;
    }

    const Rule& ruleAt(int index) const {
        return rules_list_[index];
    }

    const std::vector< Rule >& rules() const {
        return rules_list_;
    }

    std::vector< Rule >& rules() {
        return rules_list_;
    }
};

void printEquations(std::ostream& os, const RuleSet::Equations& equations);
void printStates(std::ostream& os, const RuleSet::States& states);
void printState(std::ostream& os, const RuleSet::State state);
void loadRulesFromFile(const std::string& rules_file, RuleSet& rule_set);

std::ostream& operator<<(std::ostream& os, const RuleSet::Rule& rule);

#endif /* RULE_H */

// Local Variables:
// mode: c++
// End:
