'# coding=utf-8
# Copyright 2018 The Google AI Language Team Authors and The HuggingFace Inc. team.
# Copyright (c) 2018, NVIDIA CORPORATION.  All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
""" Finetuning the library models for sequence classification on GLUE (Bert, XLM, XLNet, RoBERTa)."""

from __future__ import absolute_import, division, print_function

import argparse
import glob
import logging
import os
import random
import copy
import numpy as np
import torch
import torch.nn.functional as F
from torch import nn
from torch.utils.data import (DataLoader, RandomSampler, SequentialSampler,
                              TensorDataset)
from torch.utils.data.distributed import DistributedSampler
from tensorboardX import SummaryWriter
from tqdm import tqdm, trange

from pytorch_transformers import (WEIGHTS_NAME, BertConfig, BertModel,
                                  BertForSequenceClassification, BertTokenizer,
                                  RobertaConfig,
                                  RobertaForSequenceClassification,
                                  RobertaTokenizer, RobertaModel,
                                  XLMConfig, XLMForSequenceClassification,
                                  XLMTokenizer, XLNetConfig, XLMModel, XLNetModel,
                                  XLNetForSequenceClassification,
                                  XLNetTokenizer)


from transformers import AdamW, get_linear_schedule_with_warmup 

from transformers import (AutoTokenizer, AutoConfig, AutoModel, AutoModelForSequenceClassification)
from transformers.modeling_outputs import SequenceClassifierOutput


from utils_easyadapt import (compute_metrics, convert_examples_to_features,
                                 output_modes, processors)

from sklearn.metrics import classification_report, f1_score

logger = logging.getLogger(__name__)

ALL_MODELS = sum(
    (tuple(conf.pretrained_config_archive_map.keys()) for conf in (BertConfig, XLNetConfig, XLMConfig, RobertaConfig)),
    ())

MODEL_CLASSES = {
    'bert': (BertConfig, BertForSequenceClassification, BertModel, BertTokenizer),
    'xlnet': (XLNetConfig, XLNetForSequenceClassification, XLNetModel, XLNetTokenizer),
    'xlm': (XLMConfig, XLMForSequenceClassification, XLMModel, XLMTokenizer),
    'roberta': (RobertaConfig, RobertaForSequenceClassification, RobertaModel, RobertaTokenizer),
    'bertweet': (AutoConfig, AutoModelForSequenceClassification, AutoModel, AutoTokenizer)
}

def reweight(cls_num_list, beta=0.9999):
    '''
    Implement reweighting by effective numbers
    :param cls_num_list: a list containing # of samples of each class
    :param beta: hyper-parameter for reweighting, see paper for more details
    :return:
    '''
    per_cls_weights = None
    #############################################################################
    # TODO: reweight each class by effective numbers                            #
    #############################################################################
    n_is = np.array(cls_num_list)
    per_cls_weights = (1 - beta) / (1 - np.power(beta, n_is))
    per_cls_weights = torch.from_numpy(per_cls_weights)
    # per_cls_weights = per_cls_weights / per_cls_weights.sum() * 10
    #############################################################################
    #                              END OF YOUR CODE                             #
    #############################################################################
    return per_cls_weights

class FocalLoss(nn.Module):
    def __init__(self, weight=None, gamma=1.0, device=None):
        super(FocalLoss, self).__init__()
        assert gamma >= 0
        self.gamma = gamma
        self.weight = weight
        self.device = device

    def forward(self, input, target):
        '''
        Implement forward of focal loss
        :param input: input predictions
        :param target: labels
        :return: tensor of focal loss in scalar
        '''
        loss = None
        zi = -input
        batch_size = input.size(0)
        zi[torch.arange(batch_size), target] *= -1
        pis = torch.sigmoid(zi)
        first_term = (1-pis) ** self.gamma
        second_term = torch.log(pis)
        multipled = torch.einsum("bj,bj->b", (first_term, second_term))
        class_weights = self.weight[target].float().to(self.device)
        loss = -class_weights.dot(multipled)
        return loss

class CustomModel(nn.Module):
    def __init__(self, config, args, weights, device=None):
        super(CustomModel, self).__init__()
        if args.model_type == "bertweet":
            self.bert = AutoModel.from_pretrained(args.model_name_or_path, from_tf=bool('.ckpt' in args.model_name_or_path), config=config).to(device)
            classifier_dropout = (
            config.classifier_dropout if config.classifier_dropout is not None else config.hidden_dropout_prob
            )
        elif args.model_type == "xlm":
            self.bert = AutoModel.from_pretrained(args.model_name_or_path, from_tf=bool('.ckpt' in args.model_name_or_path),
                                            config=config).to(device)
            classifier_dropout = (
            config.classifier_dropout if config.classifier_dropout is not None else config.hidden_dropout_prob
            )
        elif args.model_type == "bert":
            self.bert = BertModel.from_pretrained(args.model_name_or_path, from_tf=bool('.ckpt' in args.model_name_or_path),
                                            config=config).to(device)
            classifier_dropout = (config.hidden_dropout_prob)
        else:
            raise Exception("Choose between bert, bertweet, or xlm models")
        

        self.weights = weights.to(device)
        self.device = device
        self.dropout = nn.Dropout(classifier_dropout).to(device)
        self.classifier = nn.Linear(3 * config.hidden_size, config.num_labels).to(device)
        self.num_labels = config.num_labels
        self.gamma = args.gamma
        self.loss = args.loss_function
        self.hidden_size = config.hidden_size

    def forward(
        self,
        input_ids=None,
        attention_mask=None,
        token_type_ids=None,
        labels=None,
        datasets=None
    ):

        outputs = self.bert(
            input_ids,
            attention_mask=attention_mask,
            token_type_ids=token_type_ids
        )
        pooled_output = outputs[1]
        triped_output = torch.cat((pooled_output, pooled_output, pooled_output), 1)
        mask = np.ones((len(datasets), 3))

        a1 = np.arange(len(datasets)) * 3 + datasets.cpu().numpy()
        np.put(mask, a1, 0)
        mask = np.repeat(mask[:, :, np.newaxis], self.hidden_size, axis=2)
        mask = torch.tensor(mask.reshape(len(datasets), -1)).to(self.device)
        triped_output = torch.mul(triped_output, mask).float()

        triped_output = self.dropout(triped_output)
        logits = self.classifier(triped_output)

        if type(outputs) is tuple:
            hidden_states = outputs[2]
            attentions = None
        else:
            hidden_states = outputs.hidden_states
            attentions = outputs.attentions


        if labels is not None:
            if self.loss == "focal_loss":
                loss_fct = FocalLoss(weight=self.weights, gamma=self.gamma, device=self.device)
            elif self.loss == "weighted_ce":
                loss_fct = nn.CrossEntropyLoss(weight=self.weights)
            elif self.loss == "ce":
                loss_fct = nn.CrossEntropyLoss()
            else:
                raise Exception("Choose a valid loss function")
            loss = loss_fct(logits.view(-1, self.num_labels), labels.view(-1))
            return SequenceClassifierOutput(
                loss=loss,
                logits=logits,
                hidden_states=hidden_states,
                attentions=attentions,
            )
        else:
            return SequenceClassifierOutput(
                loss=None,
                logits=logits,
                hidden_states=outputs.hidden_states,
                attentions=outputs.attentions,
            )
    
def set_seed(args):
    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    if args.n_gpu > 0:
        torch.cuda.manual_seed_all(args.seed)


def train(args, train_dataset, model, tokenizer):
    """ Train the model """
    if args.local_rank in [-1, 0]:
        tb_writer = SummaryWriter()

    args.train_batch_size = args.per_gpu_train_batch_size * max(1, args.n_gpu)
    train_sampler = RandomSampler(train_dataset) if args.local_rank == -1 else DistributedSampler(train_dataset)
    train_dataloader = DataLoader(train_dataset, sampler=train_sampler, batch_size=args.train_batch_size)

    if args.max_steps > 0:
        t_total = args.max_steps
        args.num_train_epochs = args.max_steps // (len(train_dataloader) // args.gradient_accumulation_steps) + 1
    else:
        t_total = len(train_dataloader) // args.gradient_accumulation_steps * args.num_train_epochs

    # Prepare optimizer and schedule (linear warmup and decay)
    no_decay = ['bias', 'LayerNorm.weight']
    optimizer_grouped_parameters = [
        {'params': [p for n, p in model.named_parameters() if not any(nd in n for nd in no_decay)],
         'weight_decay': args.weight_decay},
        {'params': [p for n, p in model.named_parameters() if any(nd in n for nd in no_decay)], 'weight_decay': 0.0}
    ]
    optimizer = AdamW(optimizer_grouped_parameters, lr=args.learning_rate, eps=args.adam_epsilon)
    scheduler = get_linear_schedule_with_warmup(optimizer, num_warmup_steps=args.warmup_steps, num_training_steps=t_total)
    if args.fp16:
        try:
            from apex import amp
        except ImportError:
            raise ImportError("Please install apex from https://www.github.com/nvidia/apex to use fp16 training.")
        model, optimizer = amp.initialize(model, optimizer, opt_level=args.fp16_opt_level)

    # multi-gpu training (should be after apex fp16 initialization)
    if args.n_gpu > 1:
        model = torch.nn.DataParallel(model)

    # Distributed training (should be after apex fp16 initialization)
    if args.local_rank != -1:
        model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.local_rank],
                                                          output_device=args.local_rank,
                                                          find_unused_parameters=True)

    # Train!
    logger.info("***** Running training *****")
    logger.info("  Num examples = %d", len(train_dataset))
    logger.info("  Num Epochs = %d", args.num_train_epochs)
    logger.info("  Instantaneous batch size per GPU = %d", args.per_gpu_train_batch_size)
    logger.info("  Total train batch size (w. parallel, distributed & accumulation) = %d",
                args.train_batch_size * args.gradient_accumulation_steps * (
                    torch.distributed.get_world_size() if args.local_rank != -1 else 1))
    logger.info("  Gradient Accumulation steps = %d", args.gradient_accumulation_steps)
    logger.info("  Total optimization steps = %d", t_total)

    global_step = 0
    tr_loss, logging_loss = 0.0, 0.0
    model.zero_grad()
    max_f1 = 0
    best_output_dir = "None"
    train_iterator = trange(int(args.num_train_epochs), desc="Epoch", disable=args.local_rank not in [-1, 0])
    set_seed(args)  # Added here for reproductibility (even between python 2 and 3)
    for _ in train_iterator:
        epoch_iterator = tqdm(train_dataloader, desc="Iteration", disable=args.local_rank not in [-1, 0])
        for step, batch in enumerate(epoch_iterator):
            model.train()
            batch = tuple(t.to(args.device) for t in batch)
            inputs = {'input_ids': batch[0],
                      'attention_mask': batch[1],
                      'token_type_ids': batch[2] if args.model_type in ['bert','xlnet'] else None,
                      # XLM and RoBERTa don't use segment_ids
                      'labels': batch[3],
                      'datasets':batch[4]}
            outputs = model(**inputs)
            loss = outputs[0]  # model outputs are always tuple in pytorch-transformers (see doc)

            if args.n_gpu > 1:
                loss = loss.mean()  # mean() to average on multi-gpu parallel training
            if args.gradient_accumulation_steps > 1:
                loss = loss / args.gradient_accumulation_steps

            if args.fp16:
                with amp.scale_loss(loss, optimizer) as scaled_loss:
                    scaled_loss.backward()
                torch.nn.utils.clip_grad_norm_(amp.master_params(optimizer), args.max_grad_norm)
            else:
                loss.backward()
                torch.nn.utils.clip_grad_norm_(model.parameters(), args.max_grad_norm)

            tr_loss += loss.item()
            if (step + 1) % args.gradient_accumulation_steps == 0:
                # Update learning rate schedule
                optimizer.step()
                scheduler.step()
                model.zero_grad()
                global_step += 1

                if args.local_rank in [-1, 0] and args.logging_steps > 0 and global_step % args.logging_steps == 0:
                    # Log metrics
                    if args.local_rank == -1 and args.evaluate_during_training:  # Only evaluate when single GPU otherwise metrics may not average well
                        results, f1 = evaluate(args, model, tokenizer, global_step, evaluate='dev')
                        for key, value in results.items():
                            tb_writer.add_scalar('eval_{}'.format(key), value, global_step)
                    tb_writer.add_scalar('lr', scheduler.get_lr()[0], global_step)
                    tb_writer.add_scalar('loss', (tr_loss - logging_loss) / args.logging_steps, global_step)
                    logging_loss = tr_loss

                if args.local_rank in [-1, 0] and args.save_steps > 0 and global_step % args.save_steps == 0:
                    # Save model checkpoint
                    output_dir = os.path.join(args.output_dir, 'checkpoint-{}'.format(global_step))
                    if f1 > max_f1:
                        max_f1 = f1
                        best_output_dir = output_dir
                    if not os.path.exists(output_dir):
                        os.makedirs(output_dir)
                    model_to_save = model.module if hasattr(model,
                                                            'module') else model  # Take care of distributed/parallel training
                    direc = os.path.join(output_dir, "pytorch_model.bin")
                    torch.save(model_to_save.state_dict(), direc)
                    torch.save(args, os.path.join(output_dir, 'training_args.bin'))
                    logger.info("Saving model checkpoint to %s", output_dir)

            if args.max_steps > 0 and global_step > args.max_steps:
                epoch_iterator.close()
                break
        if args.max_steps > 0 and global_step > args.max_steps:
            train_iterator.close()
            break

    if args.local_rank in [-1, 0]:
        tb_writer.close()

    return global_step, tr_loss / global_step, best_output_dir


def load_and_cache_examples(args, task, tokenizer, evaluate="train"):
    if args.local_rank not in [-1, 0] and not evaluate:
        torch.distributed.barrier()  # Make sure only the first process in distributed training process the dataset, and the others will use the cache

    processor = processors[task]()
    output_mode = output_modes[task]
    # Load data features from cache or dataset file
    cached_features_file = os.path.join(args.data_dir, 'cached_{}_{}_{}_{}'.format(
        evaluate,
        list(filter(None, args.model_name_or_path.split('/'))).pop(),
        str(args.loss_function),
        str(args.max_seq_length),
        str(task)))
    if os.path.exists(cached_features_file):  # os.path.exists(cached_features_file)
        logger.info("Loading features from cached file %s", cached_features_file)
        features_and_class_weights = torch.load(cached_features_file)
        features = features_and_class_weights["features"]
        per_cls_weights = features_and_class_weights["weights"]
    else:
        logger.info("Creating features from dataset file at %s", args.data_dir)
        label_list = processor.get_labels()
        label_map = {label : i for i, label in enumerate(label_list)}

        if task in ['mnli', 'mnli-mm'] and args.model_type in ['roberta']:
            # HACK(label indices are swapped in RoBERTa pretrained model)
            label_list[1], label_list[2] = label_list[2], label_list[1]

        if evaluate == 'train':
            examples, cls_num = processor.get_train_examples(args.data_dir)
            if args.loss_function == "focal_loss":
                per_cls_weights = reweight(cls_num, beta=args.beta)
            elif args.loss_function == "weighted_ce":
                per_cls_weights = [label_map[e.label] for e in examples]
            else:
                per_cls_weights = 0
        elif evaluate == 'dev':
            examples = processor.get_dev_examples(args.data_dir)
            per_cls_weights = 0
        elif evaluate == 'test':
            examples = processor.get_test_examples(args.data_dir)
            per_cls_weights = 0
        elif evaluate == 'context':
            examples = processor.get_context_examples(args.data_dir)
            per_cls_weights = 0
        elif evaluate == 'standalone':
            examples = processor.get_standalone_examples(args.data_dir)
            per_cls_weights = 0
        # examples = processor.get_test_examples(args.data_dir) if evaluate else processor.get_train_examples(
        #     args.data_dir)
        features = convert_examples_to_features(examples, label_list, args.max_seq_length, tokenizer, output_mode,
                                                cls_token_at_end=bool(args.model_type in ['xlnet']),
                                                # xlnet has a cls token at the end
                                                cls_token=tokenizer.cls_token,
                                                cls_token_segment_id=2 if args.model_type in ['xlnet'] else 0,
                                                sep_token=tokenizer.sep_token,
                                                sep_token_extra=bool(args.model_type in ['roberta']),
                                                # roberta uses an extra separator b/w pairs of sentences, cf. github.com/pytorch/fairseq/commit/1684e166e3da03f5b600dbb7855cb98ddfcd0805
                                                pad_on_left=bool(args.model_type in ['xlnet']),
                                                # pad on the left for xlnet
                                                pad_token=tokenizer.convert_tokens_to_ids([tokenizer.pad_token])[0],
                                                pad_token_segment_id=4 if args.model_type in ['xlnet'] else 0,
                                                )
        features_and_class_weights = {"weights": per_cls_weights, "features": features}
        if args.local_rank in [-1, 0]:
            logger.info("Saving features into cached file %s", cached_features_file)
            #   torch.save(per_cls_weights, cached_features_file)
            torch.save(features_and_class_weights, cached_features_file)

    if args.local_rank == 0 and not evaluate:
        torch.distributed.barrier()  # Make sure only the first process in distributed training process the dataset, and the others will use the cache


    # subset_length = int(len(features) * 5 // 100)
    # # print(subset_length)
    # features = random.sample(features, subset_length)


    # Get a subset of the dataset to do quick experiment, random_subsample is percent
    if args.random_subsample != 100 and evaluate == 'train': # and evaluate == False
        # random_subsample = 5
        subset_length = int(len(features) * args.random_subsample // 100)
        # print(subset_length)
        features = random.sample(features, subset_length)
        print("AA, important! here I do random sample!")

    # Convert to Tensors and build dataset
    all_input_ids = torch.tensor([f.input_ids for f in features], dtype=torch.long)
    all_input_mask = torch.tensor([f.input_mask for f in features], dtype=torch.long)
    all_segment_ids = torch.tensor([f.segment_ids for f in features], dtype=torch.long)
    if output_mode == "classification":
        all_label_ids = torch.tensor([f.label_id for f in features], dtype=torch.long)
    elif output_mode == "regression":
        all_label_ids = torch.tensor([f.label_id for f in features], dtype=torch.float)
    
    # for f in features:
    #     print(f.dataset)
    data_set = torch.tensor([f.dataset for f in features], dtype=torch.long)   


    dataset = TensorDataset(all_input_ids, all_input_mask, all_segment_ids, all_label_ids, data_set)

    if evaluate == 'train':
        return dataset, per_cls_weights
    return dataset

def evaluate(args, model, tokenizer, prefix="", evaluate = 'train'):
    # Loop to handle MNLI double evaluation (matched, mis-matched)
    eval_task_names = ("mnli", "mnli-mm") if args.task_name == "mnli" else (args.task_name,)
    eval_outputs_dirs = (args.output_dir, args.output_dir + '-MM') if args.task_name == "mnli" else (args.output_dir,)

    results = {}
    for eval_task, eval_output_dir in zip(eval_task_names, eval_outputs_dirs):
        eval_dataset = load_and_cache_examples(args, eval_task, tokenizer, evaluate=evaluate)

        if not os.path.exists(eval_output_dir) and args.local_rank in [-1, 0]:
            os.makedirs(eval_output_dir)

        args.eval_batch_size = args.per_gpu_eval_batch_size * max(1, args.n_gpu)
        # Note that DistributedSampler samples randomly
        eval_sampler = SequentialSampler(eval_dataset) if args.local_rank == -1 else DistributedSampler(eval_dataset)
        eval_dataloader = DataLoader(eval_dataset, sampler=eval_sampler, batch_size=args.eval_batch_size)

        # Eval!
        logger.info("***** Running evaluation {} *****".format(prefix))
        logger.info("  Num examples = %d", len(eval_dataset))
        logger.info("  Batch size = %d", args.eval_batch_size)
        eval_loss = 0.0
        nb_eval_steps = 0
        preds = None
        out_label_ids = None
        for batch in tqdm(eval_dataloader, desc="Evaluating"):
            model.eval()
            batch = tuple(t.to(args.device) for t in batch)

            with torch.no_grad():
                inputs = {'input_ids': batch[0],
                          'attention_mask': batch[1],
                          'token_type_ids': batch[2] if args.model_type in ['bert', 'xlnet'] else None,
                          # XLM and RoBERTa don't use segment_ids
                          'labels': batch[3],
                          'datasets': batch[4]}
                # print(inputs["input_ids"].shape, inputs["attention_mask"].shape, inputs["token_type_ids"].shape, inputs["labels"].shape)
                outputs = model(input_ids = inputs["input_ids"], \
                                attention_mask = inputs["attention_mask"], \
                                token_type_ids = inputs["token_type_ids"], \
                                labels = inputs["labels"], \
                                datasets = inputs["datasets"], \
                                )
                # outputs = model(**inputs)
                # print(outputs)
                
                tmp_eval_loss, logits = outputs[:2]
                # print(tmp_eval_loss)

                eval_loss += tmp_eval_loss.mean().item()
            nb_eval_steps += 1
            if preds is None:
                preds = logits.detach().cpu().numpy()
                out_label_ids = inputs['labels'].detach().cpu().numpy()
            else:
                preds = np.append(preds, logits.detach().cpu().numpy(), axis=0)
                out_label_ids = np.append(out_label_ids, inputs['labels'].detach().cpu().numpy(), axis=0)

            # if change output result per questions, need to change 4 places
            if len(preds) % 1000000 == 0:
                output_pred_file = os.path.join(eval_output_dir, "pred_results_{}.txt".format(len(preds) // 1000000))
                with open(output_pred_file, "w") as writer:
                    logger.info(
                        "***** Saved the {} * {} pred results {} *****".format(len(preds) // 1000000, 1000000, prefix))
                    for r in np.argmax(preds, axis=1):
                        writer.write("{}\n".format(r))

            if len(preds) % 1000000 == 0:
                output_pred_logit_file = os.path.join(eval_output_dir, "pred_logits_{}.txt".format(len(preds) // 1000000))
                with open(output_pred_logit_file, "w") as writer:
                    logger.info(
                        "***** Saved the {} * {} pred logits {} *****".format(len(preds) // 1000000, 1000000, prefix))
                    for r in preds:
                        writer.write("\t".join([str(rr) for rr in r]) + "\n")

        pred_logits = copy.deepcopy(preds)
        eval_loss = eval_loss / nb_eval_steps
        if args.output_mode == "classification":
            preds = np.argmax(preds, axis=1)
        elif args.output_mode == "regression":
            preds = np.squeeze(preds)

        if args.max_f1:
            result = compute_metrics("newsela_task2_max_f1", pred_logits, out_label_ids)
        elif args.max_f1_with_threshold:
            result = compute_metrics("newsela_task2_max_f1_with_threshold", pred_logits,
                                     [out_label_ids, args.predefined_threshold])
        # elif args.max_f1_with_threshold and args.task_name == 'newsela_task1':
        #     result = compute_metrics("newsela_task2_max_f1_with_threshold", pred_logits,
        #                              [out_label_ids, args.predefined_threshold])
        else:
            result = compute_metrics(eval_task, preds, out_label_ids)
        results.update(result)

        f1 = f1_score(out_label_ids, preds, average="macro")
            
        print(classification_report(out_label_ids, preds, target_names=["Supporting", "Refuting", "Other"], digits=4))

        output_eval_file = os.path.join(eval_output_dir, "eval_results_{}.txt".format(prefix))
        with open(output_eval_file, "w") as writer:
            logger.info("***** Eval results {} *****".format(prefix))
            for key in sorted(result.keys()):
                logger.info("  %s = %s", key, str(result[key]))
                writer.write("%s = %s\n" % (key, str(result[key])))

        output_pred_file = os.path.join(eval_output_dir, "pred_results_{}.txt".format(prefix))
        with open(output_pred_file, "w") as writer:
            logger.info("***** Pred results {} *****".format(prefix))
            for r in preds:
                writer.write("{}\n".format(r))

        output_pred_logit_file = os.path.join(eval_output_dir, "pred_logits_{}.txt".format(prefix))
        with open(output_pred_logit_file, "w") as writer:
            logger.info("***** Pred logits {} *****".format(prefix))
            for r in pred_logits:
                writer.write("\t".join([str(rr) for rr in r]) + "\n")

    return results, f1




def main():
    parser = argparse.ArgumentParser()

    ## Required parameters
    parser.add_argument("--data_dir", default=None, type=str, required=True,
                        help="The input data dir. Should contain the .tsv files (or other data files) for the task.")
    parser.add_argument("--model_type", default=None, type=str, required=True,
                        help="Model type selected in the list: " + ", ".join(MODEL_CLASSES.keys()))
    parser.add_argument("--model_name_or_path", default=None, type=str, required=True,
                        help="Path to pre-trained model or shortcut name selected in the list: " + ", ".join(
                            ALL_MODELS))
    parser.add_argument("--task_name", default=None, type=str, required=True,
                        help="The name of the task to train selected in the list: " + ", ".join(processors.keys()))
    parser.add_argument("--output_dir", default=None, type=str, required=True,
                        help="The output directory where the model predictions and checkpoints will be written.")

    ## Other parameters
    parser.add_argument("--config_name", default="", type=str,
                        help="Pretrained config name or path if not the same as model_name")
    parser.add_argument("--tokenizer_name", default="", type=str,
                        help="Pretrained tokenizer name or path if not the same as model_name")
    parser.add_argument("--cache_dir", default="", type=str,
                        help="Where do you want to store the pre-trained models downloaded from s3")
    parser.add_argument("--max_seq_length", default=128, type=int,
                        help="The maximum total input sequence length after tokenization. Sequences longer "
                             "than this will be truncated, sequences shorter will be padded.")
    parser.add_argument("--do_train", action='store_true',
                        help="Whether to run training.")
    parser.add_argument("--max_f1", action='store_true',
                        help="Compute max F1.")
    parser.add_argument("--max_f1_with_threshold", action='store_true',
                        help="Compute max F1 and F1 using given threshold.")
    parser.add_argument("--predefined_threshold", type=float,
                        help="using with max_f1_with_threshold, will compute F1 using this threshold")
    parser.add_argument("--do_eval", action='store_true',
                        help="Whether to run eval on the dev set.")
    parser.add_argument("--evaluate_during_training", action='store_true',
                        help="Rul evaluation during training at each logging step.")
    parser.add_argument("--do_lower_case", action='store_true',
                        help="Set this flag if you are using an uncased model.")
    parser.add_argument("--gamma", default=1.0, type=float)

    parser.add_argument("--beta", default=0.9999, type=float)
    
    parser.add_argument("--per_gpu_train_batch_size", default=8, type=int,
                        help="Batch size per GPU/CPU for training.")
    parser.add_argument("--per_gpu_eval_batch_size", default=8, type=int,
                        help="Batch size per GPU/CPU for evaluation.")
    parser.add_argument('--gradient_accumulation_steps', type=int, default=1,
                        help="Number of updates steps to accumulate before performing a backward/update pass.")
    parser.add_argument("--learning_rate", default=5e-5, type=float,
                        help="The initial learning rate for Adam.")
    parser.add_argument("--weight_decay", default=0.0, type=float,
                        help="Weight deay if we apply some.")
    parser.add_argument("--adam_epsilon", default=1e-8, type=float,
                        help="Epsilon for Adam optimizer.")
    parser.add_argument("--max_grad_norm", default=1.0, type=float,
                        help="Max gradient norm.")
    parser.add_argument("--num_train_epochs", default=3.0, type=float,
                        help="Total number of training epochs to perform.")
    parser.add_argument("--max_steps", default=-1, type=int,
                        help="If > 0: set total number of training steps to perform. Override num_train_epochs.")
    parser.add_argument("--warmup_steps", default=0, type=int,
                        help="Linear warmup over warmup_steps.")
    parser.add_argument("--model_size", default="base", type=str,
                        help="Linear warmup over warmup_steps.")
    parser.add_argument("--loss_function", default="cross_entropy", type=str)

    parser.add_argument('--logging_steps', type=int, default=50,
                        help="Log every X updates steps.")
    parser.add_argument('--save_steps', type=int, default=50000000000,  # default is 50
                        help="Save checkpoint every X updates steps.")
    parser.add_argument("--eval_all_checkpoints", action='store_true',
                        help="Evaluate all checkpoints starting with the same prefix as model_name ending and ending with step number")
    parser.add_argument("--no_cuda", action='store_true',
                        help="Avoid using CUDA when available")
    parser.add_argument('--overwrite_output_dir', action='store_true',
                        help="Overwrite the content of the output directory")
    parser.add_argument('--overwrite_cache', action='store_true',
                        help="Overwrite the cached training and evaluation sets")
    parser.add_argument('--seed', type=int, default=42,
                        help="random seed for initialization")
    parser.add_argument('--fp16', action='store_true',
                        help="Whether to use 16-bit (mixed) precision (through NVIDIA apex) instead of 32-bit")
    parser.add_argument('--fp16_opt_level', type=str, default='O1',
                        help="For fp16: Apex AMP optimization level selected in ['O0', 'O1', 'O2', and 'O3']."
                             "See details at https://nvidia.github.io/apex/amp.html")
    parser.add_argument("--local_rank", type=int, default=-1,
                        help="For distributed training: local_rank")
    parser.add_argument('--server_ip', type=str, default='', help="For distant debugging.")
    parser.add_argument('--server_port', type=str, default='', help="For distant debugging.")
    parser.add_argument('--random_subsample', type=float, default=100, help="Random subsample a percent of train/test data to quickly run experiment.")
    parser.add_argument('--randomlize_pooler_layer', action='store_true',
                        help="Randomlize the pooler within the BertModel")

    args = parser.parse_args()

    if os.path.exists(args.output_dir) and os.listdir(
            args.output_dir) and args.do_train and not args.overwrite_output_dir:
        raise ValueError(
            "Output directory ({}) already exists and is not empty. Use --overwrite_output_dir to overcome.".format(
                args.output_dir))

    # Setup distant debugging if needed
    if args.server_ip and args.server_port:
        # Distant debugging - see https://code.visualstudio.com/docs/python/debugging#_attach-to-a-local-script
        import ptvsd
        print("Waiting for debugger attach")
        ptvsd.enable_attach(address=(args.server_ip, args.server_port), redirect_output=True)
        ptvsd.wait_for_attach()

    # Setup CUDA, GPU & distributed training
    if args.local_rank == -1 or args.no_cuda:
        device = torch.device("cuda" if torch.cuda.is_available() and not args.no_cuda else "cpu")
        args.n_gpu = torch.cuda.device_count()
    else:  # Initializes the distributed backend which will take care of sychronizing nodes/GPUs
        torch.cuda.set_device(args.local_rank)
        device = torch.device("cuda", args.local_rank)
        torch.distributed.init_process_group(backend='nccl')
        args.n_gpu = 1
    args.device = device

    # Setup logging
    logging.basicConfig(format='%(asctime)s - %(levelname)s - %(name)s -   %(message)s',
                        datefmt='%m/%d/%Y %H:%M:%S',
                        level=logging.INFO if args.local_rank in [-1, 0] else logging.WARN)
    logger.warning("Process rank: %s, device: %s, n_gpu: %s, distributed training: %s, 16-bits training: %s",
                   args.local_rank, device, args.n_gpu, bool(args.local_rank != -1), args.fp16)

    # Set seed
    set_seed(args)

    # Prepare GLUE task
    args.task_name = args.task_name.lower()
    if args.task_name not in processors:
        raise ValueError("Task not found: %s" % (args.task_name))
    processor = processors[args.task_name]()
    args.output_mode = output_modes[args.task_name]
    label_list = processor.get_labels()
    num_labels = len(label_list)

    # Load pretrained model and tokenizer
        # Training


    if args.local_rank not in [-1, 0]:
        torch.distributed.barrier()  # Make sure only the first process in distributed training will download model & vocab

    args.model_type = args.model_type.lower()
    config_class, model_class, model_gen, tokenizer_class = MODEL_CLASSES[args.model_type]
    tokenizer = tokenizer_class.from_pretrained(args.tokenizer_name if args.tokenizer_name else args.model_name_or_path,
                                                    do_lower_case=args.do_lower_case, normalization=True)
    config = config_class.from_pretrained(args.config_name if args.config_name else args.model_name_or_path,
                                            num_labels=num_labels, finetuning_task=args.task_name, output_hidden_states=True)

    if args.do_train:
        train_dataset, per_cls_weights = load_and_cache_examples(args, args.task_name, tokenizer, evaluate='train')

    if args.loss_function == "weighted_ce":
        class_weights = get_class_weights(per_cls_weights, num_labels)
        model = CustomModel(config=config, weights=class_weights, args=args, device=args.device)
    elif args.loss_function == "focal_loss":
        model = CustomModel(config=config, weights=per_cls_weights, args=args, device=args.device)
    elif args.loss_function == "ce":
        model = CustomModel(config=config, weights=None, args=args, device=args.device)    

    if args.do_train:
        global_step, tr_loss, best_directory = train(args, train_dataset, model, tokenizer)
        print(best_directory)
        logger.info(" global_step = %s, average loss = %s", global_step, tr_loss)
    if args.randomlize_pooler_layer == True:
        print("will randomlize pooler layer")
        getattr(getattr(getattr(model, "bert"), "pooler"), "dense").weight.data.normal_(mean=0.0, std=0.02)
        getattr(getattr(getattr(model, "bert"), "pooler"), "dense").bias.data.zero_()

    if args.local_rank == 0:
        torch.distributed.barrier()  # Make sure only the first process in distributed training will download model & vocab

    model.to(args.device)

    logger.info("Training/evaluation parameters %s", args)

    # Saving best-practices: if you use defaults names for the model, you can reload it using from_pretrained()
    if args.do_train and (args.local_rank == -1 or torch.distributed.get_rank() == 0):
        # Create output directory if needed
        if not os.path.exists(args.output_dir) and args.local_rank in [-1, 0]:
            os.makedirs(args.output_dir)

        logger.info("Saving model checkpoint to %s", args.output_dir)
        # Save a trained model, configuration and tokenizer using `save_pretrained()`.
        # They can then be reloaded using `from_pretrained()`

        # Load a trained model and vocabulary that you have fine-tuned
        direc = os.path.join(best_directory, "pytorch_model.bin")
        tokenizer.save_pretrained(args.output_dir)

        if args.loss_function == "focal_loss":
            model = CustomModel(config=config, weights=per_cls_weights, args=args, device=args.device)
        elif args.loss_function == "weighted_ce":
            model = CustomModel(config=config, weights=class_weights, args=args, device=args.device)
        elif args.loss_function == "ce":
            model = CustomModel(config=config, weights=None, args=args, device=args.device)

        model.load_state_dict(torch.load(direc))
        tokenizer = tokenizer_class.from_pretrained(args.output_dir, do_lower_case=args.do_lower_case, normalization=True)
        model.to(args.device)

        model_to_save = model.module if hasattr(model,
                                                'module') else model
        direc = os.path.join(args.output_dir, "pytorch_model.bin")
        torch.save(model_to_save.state_dict(), direc)
        
    # Evaluation
    results = {}
    if args.do_eval and args.local_rank in [-1, 0]:
        tokenizer = tokenizer_class.from_pretrained(args.output_dir, do_lower_case=args.do_lower_case, normalization=True)
        checkpoints = [args.output_dir]
        if args.eval_all_checkpoints:
            checkpoints = list(
                os.path.dirname(c) for c in sorted(glob.glob(args.output_dir + '/**/' + WEIGHTS_NAME, recursive=True)))
            logging.getLogger("pytorch_transformers.modeling_utils").setLevel(logging.WARN)  # Reduce logging

            # checkpoints = ['../log/Newsela_train_dev_test_all_task2/checkpoint-283926']

        logger.info("Evaluate the following checkpoints: %s", checkpoints)

        for checkpoint in checkpoints:
            global_step = checkpoint.split('-')[-1] if len(checkpoints) > 1 else ""

            if args.loss_function == "focal_loss":
                model = CustomModel(config=config, weights=per_cls_weights, args=args, device=args.device)
            elif args.loss_function == "weighted_ce":
                model = CustomModel(config=config, weights=class_weights, args=args, device=args.device)
            else:
                model = CustomModel(config=config, weights=None, args=args, device=args.device)
            
            direc = os.path.join(checkpoint, "pytorch_model.bin")
            model.load_state_dict(torch.load(direc))
            model.to(args.device)
            result, f1 = evaluate(args, model, tokenizer, prefix=global_step, evaluate = 'test')
            result = dict((k + '_{}'.format(global_step), v) for k, v in result.items())
            results.update(result)

        return results


if __name__ == "__main__":
    main()
