import React, { Component } from 'react';
import { withTracker } from 'meteor/react-meteor-data';
import { Tracker } from 'meteor/tracker'
import { Redirect } from "react-router-dom";

import { Annotations, Assignments, Progress } from '../api/documents';
import { gup } from '../util/gup.js';

export class Assign extends Component {
  constructor(props) {
    super(props);

    this.setAssignment = this.setAssignment.bind(this);

    this.state = {
      assignment: "",
      tutorial: "",
    };
  }

  setAssignment() {
    // This is somewhat complicated because we need to consider the possibility
    // that HITs were returned (and so some of the assignments are essentially
    // abandoned). We don't know for sure when something is returned, so we (1)
    // find completed assignments, which we know to ignore (2) go through
    // things that were assigned, but not completed in order from the most
    // recent, counting down the assignments, taking the last one to reach
    // zero. Effectively this means assuming the oldest cases are most likely
    // to be abandoned.
    if (! this.props.areReady) {
      // Need to wait
    } else if (this.state.assignment == "") {
      if (gup("assignmentId") == "ASSIGNMENT_ID_NOT_AVAILABLE") {
        const proposed_assignment_id = gup("constrainAssignment");
        const permitted_assignments = this.props.assignments.filter(assignment => {
          return assignment.group == this.props.match.params.group &&
            (proposed_assignment_id == '' || proposed_assignment_id == assignment._id);
        })

        if (permitted_assignments.length > 0) {
          Meteor.call('assignments.preview', permitted_assignments[0]._id, this.props.match.params.group);
          this.setState({
            assignment: permitted_assignments[0]._id,
            tutorial: "",
          });
        } else {
          Meteor.call('assignments.preview', 'no-assignment-available', this.props.match.params.group);
          this.setState({
            assignment: "none-available",
            tutorial: "",
          });
        }
        return;
      }

      const user = gup("workerId");
      const proposed_assignment_id = gup("constrainAssignment");

      // Get set of assignments this worker has been assigned
      const previously_assigned = new Set(
        this.props.progress.filter(report => {
          return report.user == user && report.assignment != undefined;
        }).map(report => {
          return report.assignment;
        })
      );
      const previously_done = new Set(
        this.props.progress.filter(report => {
          return report.user == user && report.assignment != undefined && report.completed;
        }).map(report => {
          return report.assignment;
        })
      );
      const permitted_assignments = new Set(
        this.props.assignments.filter(assignment => {
          return assignment.group == this.props.match.params.group &&
            (! previously_done.has(assignment._id)) &&
            (proposed_assignment_id == '' || proposed_assignment_id == assignment._id);
        }).map(assignment => {
          return assignment._id
        })
      );

      // Get counts of completed assignments
      const completed = this.props.progress.reduce((counts, report) => {
        if (
          report.assignment != undefined &&
          report.completed && 
          permitted_assignments.has(report.assignment)
        ) {
          counts[report.assignment] = (counts[report.assignment] || 0) + 1;
        }
        return counts;
      }, {});
      // Get counts of incomplete assignments
      var non_empty = false;
      const incomplete = this.props.assignments.reduce((counts, assignment) => {
        if (permitted_assignments.has(assignment._id)) {
          const left = assignment.count - (completed[assignment._id] || 0);
          if (left > 0) {
            counts[assignment._id] = left;
            non_empty = true;
          }
        }
        return counts;
      }, {});

      if (non_empty) {
        // Get the assignments currently in progress and permitted, sorted by
        // time (newest first)
        const available = this.props.progress.filter(report => {
          return report.assignment != undefined &&
            (! report.completed) &&
            permitted_assignments.has(report.assignment);
        });
        available.sort((a, b) => {
          return a.assigned > b.assigned ? -1 : 1;
        });

        // Find the oldest assignment still available
        var last = "";
        available.forEach(report => {
          incomplete[report.assignment] -= 1;
          if (incomplete[report.assignment] == 0) {
            last = report.assignment;
          }
        });
        // Also check the full set of assignments because we may have some not
        // assigned at all
        this.props.assignments.forEach(assignment => {
          if (incomplete[assignment._id] > 0 &&
              permitted_assignments.has(assignment._id)
          ) {
            last = assignment._id;
          }
        });

        // Get the assignment and assign it
        const assignment = this.props.assignments.find(assignment => {
          return assignment._id == last;
        });
        // Get the set of tutorials this worker has previously completed
        const tutorials = new Set(
          this.props.progress.filter(report => {
            return report.tutorial != undefined && 
              report.completed &&
              report.user == user;
          }).map(report => {
            return report.tutorial;
          })
        );
        const anno = this.props.annotations.find(anno => {
          return anno._id == assignment.ann_id;
        });
        if (anno != undefined) {
          const clusters = anno.clusters;
          if (! previously_assigned.has(last)) {
            Meteor.call('assignments.assign', assignment._id, assignment.text_id, user, clusters, anno.labels, anno.checks, anno.focus_mention);
          }
          if (assignment.tutorial != undefined && (! tutorials.has(assignment.tutorial))) {
            this.setState({
              assignment: assignment._id,
              tutorial: assignment.tutorial,
            });
          } else {
            this.setState({
              assignment: assignment._id,
              tutorial: "",
            });
          }
        }
      } else if (this.state.assignment != "none-available") {
        const user = gup("workerId");
        Meteor.call('logError', user, "Error: No assignments available for "+ this.props.location.pathname + this.props.location.search);
        this.setState({
          assignment: "none-available",
        });
      }
    }
  }

  componentDidUpdate() {
    if (this.props.assignments.length > 0 && this.props.annotations.length > 0) {
      this.setAssignment();
    }
  }

  componentDidMount() {
    if (this.props.assignments.length > 0 && this.props.annotations.length > 0) {
      this.setAssignment();
    }
  }

  render() {
    const assignment_id = gup("internalAssignment");
    if (assignment_id != "") {
      // When run with a specific assignment (not actually assigning, but jumping straight to doing it)
      const todo = this.props.assignments.find(assignment => {
        return assignment._id == assignment_id;
      });
      if (todo != undefined) {
        const target = "/annotate/" +
          todo.ui +
          this.props.location.search.split("&internalAssignment=").join("").split("&textId=").join("").split("&instructions=").join("") +
          "&instructions=" + todo.instructions +
          "&textId=" + todo.text_id +
          "&internalAssignment=" + todo._id;
        return (
          <Redirect to={target} />
        );
      } else {
        return (
          <div>Loading...</div>
        );
      }
    } else if (this.state.assignment == "none-available") {
      // When we have determined there is no suitable task for this user go to a page that says there is no task
      const target = "/no-task/" + this.props.location.search;
      return (
          <Redirect to={target} />
      );
    } else if (this.state.assignment == "") {
      // When the assignment information is still loading across the network
      return (
        <div>Loading...</div>
      );
    } else if (this.state.tutorial != "") {
      // When we have an assignment and there is an associated tutorial
      const target = "/tutorial/" +
        this.state.tutorial + 
        this.props.location.search.split("&internalAssignment=").join("").split("&textId=").join("").split("&instructions=").join("") +
        "&group="+ this.props.match.params.group +
        "&internalAssignment=" + this.state.assignment;
      return (
          <Redirect to={target} />
      );
    } else {
      // When we have an assignment and there is no tutorial
      const todo = this.props.assignments.find(assignment => {
        return assignment._id == this.state.assignment;
      });
      const target = 
        (gup("assignmentId") == "ASSIGNMENT_ID_NOT_AVAILABLE" ? "/preview/" : "/annotate/") +
        todo.ui +
        this.props.location.search.split("&internalAssignment=").join("").split("&textId=").join("").split("&instructions=").join("") +
        "&instructions=" + todo.instructions +
        "&textId=" + todo.text_id +
        "&internalAssignment=" + todo._id;
      return (
          <Redirect to={target} />
      );
    }
  }
}

export default AssignContainer = withTracker(() => {
  const handles = [
    Meteor.subscribe('assignments-all'),
    Meteor.subscribe('progress-all'),
    Meteor.subscribe('annotations-user', "server"),
  ];

  return {
    annotations: Annotations.find().fetch(),
    assignments: Assignments.find().fetch(),
    progress: Progress.find().fetch(),
    areReady: handles.every(handle => handle.ready()),
  };
})(Assign);
