# Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.  
# SPDX-License-Identifier: CC-BY-NC-4.0
import argparse
import json
from collections import defaultdict
from transformers import AutoTokenizer

import numpy as np

from tqdm import tqdm

INITIAL_MAX_LENGTH = 16


def main():
    parser = argparse.ArgumentParser(description="Calculate token distributions in a blink dataset")
    parser.add_argument("--input", help="Input blink file", required=True)
    parser.add_argument("--tokenizer", help="Model to use for tokenization", required=True)
    parser.add_argument("--debug", help="Enable debug", action="store_true")
    args = parser.parse_args()

    tokenizer = AutoTokenizer.from_pretrained(args.tokenizer)

    debug_messages_left = 10
    distributions = defaultdict(lambda: np.zeros(INITIAL_MAX_LENGTH))
    with open(args.input, 'r') as reader:
        for line in tqdm(reader, desc="Processing dataset"):
            sample = json.loads(line)
            for field in ["context_left", "context_right", "label_title", "label", "mention"]:
                lengths = distributions[field]
                tokenized_field = tokenizer.tokenize(sample[field])
                if args.debug and debug_messages_left > 0:
                    print("Field: %s. Raw input: %s. Tokenized input: %s" % (field, sample[field], str(tokenized_field)))
                    debug_messages_left -= 1
                cur_length = len(tokenized_field)
                while cur_length >= len(lengths):
                    distributions[field] = np.zeros(len(lengths)*2)
                    distributions[field][:len(lengths)] = lengths
                    lengths = distributions[field]
                lengths[cur_length] += 1

    for field in distributions:
        print("Distribution for field \"%s\"" % field)
        distr_output = []
        lengths = np.cumsum(distributions[field])
        prev_percentage = 0.0
        for idx in range(len(lengths)):
            cur_percentage = float(lengths[idx])/lengths[-1]
            if cur_percentage > prev_percentage:
                prev_percentage = cur_percentage
                distr_output.append("%d: %.2f%%" % (idx, 100*cur_percentage))
        if len(distr_output) > 50:
            distr_output = distr_output[:49] + ["..."] + distr_output[-49:]
        print(", ".join(distr_output))


if __name__ == '__main__':
    main()
