#!/usr/bin/env python
# coding=utf-8
# Copyright 2021 The HuggingFace Team All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""
Fine-tuning the library's seq2seq models for question answering using the 🤗 Seq2SeqTrainer.
"""
# You can also adapt this script on your own question answering task. Pointers for this are left as comments.
import logging
import os
import sys
import random
from tqdm import tqdm

import torch
import datasets
from datasets import load_dataset, DatasetDict, concatenate_datasets, load_from_disk
import numpy as np
import transformers
from transformers import (
    AutoConfig,
    AutoModelForSeq2SeqLM,
    AutoTokenizer,
    DataCollatorForSeq2Seq,
    HfArgumentParser,
    set_seed
)
from transformers.trainer_utils import get_last_checkpoint
from src.dataset.glue_tasks import AutoTask
from src.utils import ModelArguments, DataTrainingArguments, Seq2SeqArguments, LoraArguments, \
            TASK_DEMONSTRATION_ONLY, get_dataset_sizes
from src.metrics import build_compute_metrics_fn, handle_metrics
from src.models import HyperLoRAConfig, HyperLoRAModelForPretrain, HyperLoRAModelForFinetune
from src.trainer import Seq2SeqHyperTrainer

import os
os.environ["WANDB_PROJECT"]="HyperLoRA"

logger = logging.getLogger(__name__)

def generate_lora_weights(hyperlora, demo_datasets):
    hyperlora.cuda()
    with torch.no_grad():
        input_ids, attention_mask = demo_datasets['demo_input_ids'], demo_datasets['demo_attention_mask']
        input_ids = torch.tensor(input_ids).cuda()
        attention_mask = torch.tensor(attention_mask).cuda()
        generate_lora_weights = hyperlora.generate_lora_weights(
                                        demo_input_ids=input_ids.unsqueeze(0),
                                        demo_attention_mask=attention_mask.unsqueeze(0))

    return generate_lora_weights


def main():
    # See all possible arguments in src/transformers/training_args.py
    # or by passing the --help flag to this script.
    # We now keep distinct sets of args, for a cleaner separation of concerns.

    parser = HfArgumentParser((ModelArguments, DataTrainingArguments, Seq2SeqArguments, LoraArguments))
    if len(sys.argv) == 2 and sys.argv[1].endswith(".json"):
        # If we pass only one argument to the script and it's the path to a json file,
        # let's parse it to get our arguments.
        model_args, data_args, training_args, lora_args = parser.parse_json_file(json_file=os.path.abspath(sys.argv[1]))
    else:
        model_args, data_args, training_args, lora_args = parser.parse_args_into_dataclasses()

    # update model_args

    # Setup logging
    logging.basicConfig(
        format="%(asctime)s - %(levelname)s - %(name)s - %(message)s",
        datefmt="%m/%d/%Y %H:%M:%S",
        handlers=[logging.StreamHandler(sys.stdout)],
    )

    log_level = training_args.get_process_log_level()
    logger.setLevel(log_level)
    datasets.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.enable_default_handler()
    transformers.utils.logging.enable_explicit_format()

    # training_args.local_rank = -1
    # Log on each process the small summary:
    logger.warning(
        f"Process rank: {training_args.local_rank}, device: {training_args.device}, n_gpu: {training_args.n_gpu}"
        + f", distributed training: {bool(training_args.local_rank != -1)}, 16-bits training: {training_args.fp16}"
    )
    logger.info(f"Training/evaluation parameters {training_args}")

    # Set seed before initializing model.
    set_seed(training_args.seed)

    training_args.local_rank = -1
    # read multi-task datasets
    if training_args.do_train or training_args.do_test:
        train_dataset = AutoTask.get(data_args.task, seed=training_args.seed).get_dataset(split="train",
                            n_obs=-1, add_prefix=True)
        # sample n_demonstrations from train_task_dataset
        rnd_demo_idx = random.sample(range(len(train_dataset)), data_args.n_demonstrations)
        demo_datasets = train_dataset.select(rnd_demo_idx)
        # filter sampled demostrations
        train_dataset = train_dataset.filter(lambda x: x not in rnd_demo_idx)
        train_dataset_sizes = [len(train_dataset)]

    if training_args.do_eval:
        eval_dataset = AutoTask.get(data_args.task, seed=training_args.seed).get_dataset(split="validation",
                                n_obs=-1, add_prefix=True, split_validation_test=True)
        eval_dataset_sizes = [len(eval_dataset)]

    if training_args.do_test:
        test_dataset = AutoTask.get(data_args.task, seed=training_args.seed).get_dataset(split="test",
                                n_obs=-1, add_prefix=True, split_validation_test=True)
        test_dataset_sizes = [len(test_dataset)]

    # Load pretrained model and tokenizer
    #
    # Distributed training:
    # The .from_pretrained methods guarantee that only one local process can concurrently
    # download model & vocab.
    hyperlora_config = HyperLoRAConfig.from_pretrained(
        model_args.hypelora_name_or_path, lora_rank=lora_args.lora_rank
    )
    plm_config = AutoConfig.from_pretrained(
        model_args.model_name_or_path,
        cache_dir=model_args.cache_dir,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )
    tokenizer = AutoTokenizer.from_pretrained(
        model_args.model_name_or_path,
        cache_dir=model_args.cache_dir,
        use_fast=model_args.use_fast_tokenizer,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )
    plm_model = AutoModelForSeq2SeqLM.from_pretrained(
        model_args.model_name_or_path,
        from_tf=bool(".ckpt" in model_args.model_name_or_path),
        config=plm_config,
        cache_dir=model_args.cache_dir,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
        torch_dtype=torch.bfloat16
    )

    # We resize the embeddings only when necessary to avoid index errors. If you are creating a model from scratch
    # on a small vocab and want a smaller embedding size, remove this test.
    embedding_size = plm_model.get_input_embeddings().weight.shape[0]
    if len(tokenizer) > embedding_size:
        plm_model.resize_token_embeddings(len(tokenizer))

    if plm_model.config.decoder_start_token_id is None:
        raise ValueError("Make sure that `config.decoder_start_token_id` is correctly defined")

    # task2id
    hyperlora_model = HyperLoRAModelForPretrain(config=hyperlora_config,
                                      model_args=model_args,
                                      lora_args=lora_args,
                                      encoder=plm_model.encoder,
                                      pretrain_task_names=data_args.task)

    # load pre-trained checkpoint
    if model_args.pretrain_checkpoint is not None:
        logger.info('Loading checkpoint from {}'.format(model_args.pretrain_checkpoint))
        hyperlora_model.load_state_dict(torch.load(os.path.join(model_args.pretrain_checkpoint, "pytorch_model.bin"), map_location='cpu'))

    # column_names = ["input", "output"]
    # Get the column names for input/target.
    input_column = data_args.input_column
    output_column = data_args.output_column

    # Temporarily set max_answer_length for training.
    max_answer_length = data_args.max_answer_length
    padding = "max_length" if data_args.pad_to_max_length else False

    if training_args.label_smoothing_factor > 0 and not hasattr(hyperlora_model, "prepare_decoder_input_ids_from_labels"):
        logger.warning(
            "label_smoothing is enabled but the `prepare_decoder_input_ids_from_labels` method is not defined for"
            f"`{hyperlora_model.__class__.__name__}`. This will lead to loss being calculated twice and will take up more memory"
        )

    max_seq_length = data_args.max_seq_length

    # Data collator
    label_pad_token_id = -100 if data_args.ignore_pad_token_for_loss else tokenizer.pad_token_id
    data_collator = DataCollatorForSeq2Seq(
        tokenizer,
        model=hyperlora_model,
        label_pad_token_id=label_pad_token_id,
        pad_to_multiple_of=8 if training_args.fp16 else None
    )

    def preprocess_demonstrations(task_demo_examples):
        demonstration_inputs = ""
        inputs = task_demo_examples[input_column]
        targets = task_demo_examples[output_column]
        assert len(inputs) == data_args.n_demonstrations
        for i in range(len(inputs)):
            demonstration_inputs += TASK_DEMONSTRATION_ONLY.format(input=inputs[i], target=targets[i])

        demo_tokenize_output= tokenizer(demonstration_inputs.strip(), max_length=max_seq_length*2, padding=padding, truncation=True)
        demo_output = {'demo_input_ids': demo_tokenize_output['input_ids'],
                       'demo_attention_mask': demo_tokenize_output['attention_mask']
                       }
        return demo_output

    def preprocess_function(examples):
        inputs = examples[input_column]
        targets = examples[output_column]

        model_inputs = tokenizer(inputs, max_length=max_seq_length, padding=padding, truncation=True)
        # Tokenize targets with text_target=...
        labels = tokenizer(text_target=targets, max_length=max_answer_length, padding=padding, truncation=True)

        # padding in the loss.
        if padding == "max_length" and data_args.ignore_pad_token_for_loss:
            labels["input_ids"] = [
                [(l if l != tokenizer.pad_token_id else -100) for l in label] for label in labels["input_ids"]
            ]

        model_inputs["labels"] = labels["input_ids"]
        return model_inputs

    # prepreocess demonstration
    demo_datasets = preprocess_demonstrations(demo_datasets)

    # generate lora modules
    generate_adapters = generate_lora_weights(hyperlora_model, demo_datasets)

    # define down_plm model, and inject generate weights
    model = HyperLoRAModelForFinetune(
        config=plm_config,
        model_args=model_args,
        lora_args=lora_args,
        generate_adapters=generate_adapters
    )

     # load fine-tuned checkpoint
    if not training_args.do_train and model_args.finetune_checkpoint is not None:
        logger.info('Loading checkpoint from {}'.format(model_args.finetune_checkpoint))
        model.load_state_dict(torch.load(os.path.join(model_args.finetune_checkpoint, "pytorch_model.bin"), map_location='cpu'))


    if training_args.do_train:
        column_names = train_dataset.column_names
        if data_args.max_train_samples is not None:
            # We will select sample from whole data if agument is specified
            max_train_samples = min(len(train_dataset), data_args.max_train_samples)
            train_dataset = train_dataset.select(range(max_train_samples))
        # Create train feature from dataset
        with training_args.main_process_first(desc="train dataset map pre-processing"):
            train_dataset = train_dataset.map(
                preprocess_function,
                batched=True,
                num_proc=data_args.preprocessing_num_workers,
                remove_columns=column_names,
                load_from_cache_file=not data_args.overwrite_cache,
                desc="Running tokenizer on train dataset",
            )
        if data_args.max_train_samples is not None:
            # Number of samples might increase during Feature Creation, We select only specified max samples
            max_train_samples = min(len(train_dataset), data_args.max_train_samples)
            train_dataset = train_dataset.select(range(max_train_samples))

    if training_args.do_eval:
        column_names = eval_dataset.column_names
        if data_args.max_eval_samples is not None:
            # We will select sample from whole data
            max_eval_samples = min(len(eval_dataset), data_args.max_eval_samples)
            eval_dataset = eval_dataset.select(range(max_eval_samples))
        # Validation Feature Creation
        with training_args.main_process_first(desc="validation dataset map pre-processing"):
            eval_dataset = eval_dataset.map(
                preprocess_function,
                batched=True,
                num_proc=data_args.preprocessing_num_workers,
                remove_columns=column_names,
                load_from_cache_file=not data_args.overwrite_cache,
                desc="Running tokenizer on validation dataset",
            )
        if data_args.max_eval_samples is not None:
            # During Feature creation dataset samples might increase, we will select required samples again
            max_eval_samples = min(len(eval_dataset), data_args.max_eval_samples)
            eval_dataset = eval_dataset.select(range(max_eval_samples))

    if training_args.do_test:
        column_names = test_dataset.column_names
        # Test Feature Creation
        with training_args.main_process_first(desc="test dataset map pre-processing"):
            test_dataset = test_dataset.map(
                preprocess_function,
                batched=True,
                num_proc=data_args.preprocessing_num_workers,
                remove_columns=column_names,
                load_from_cache_file=not data_args.overwrite_cache,
                desc="Running tokenizer on test dataset",
            )

    # Defines the metrics for evaluation.
    task2metric_name = {'cola': 'mcc', 'sst2': 'acc', 'mrpc': 'f1',
                        'stsb': 'pearson_corrcoef', 'qqp': 'f1', 'mnli': 'acc', 'qnli': 'acc', 'rte': 'acc'
                        }
    training_args.metric_for_best_model = task2metric_name[data_args.task]
    compute_metrics_fn = build_compute_metrics_fn([data_args.task], tokenizer)[data_args.task] if training_args.predict_with_generate else None

    # no shuffle in inter-task
    # train_dataset = train_dataset.shuffle(seed=training_args.seed)
    # call_backs = [EarlyStoppingCallback(early_stopping_patience=3)]
    # Initialize our Trainer
    trainer = Seq2SeqHyperTrainer(
        train_dataset_sizes=train_dataset_sizes if training_args.do_train else None,
        val_dataset_sizes=eval_dataset_sizes if training_args.do_eval else None,
        use_multitask_sampler=False,
        model=model,
        args=training_args,
        train_dataset=train_dataset if training_args.do_train else None,
        eval_dataset=eval_dataset if training_args.do_eval else None,
        tokenizer=tokenizer,
        data_collator=data_collator,
        compute_metrics=compute_metrics_fn if training_args.predict_with_generate else None,
        # callbacks=call_backs
    )

    # Training
    if training_args.do_train:
        checkpoint = None
        if training_args.resume_from_checkpoint is not None:
            checkpoint = training_args.resume_from_checkpoint
        train_result = trainer.train(resume_from_checkpoint=checkpoint)
        # trainer.save_model()  # Saves the tokenizer too for easy upload

        metrics = train_result.metrics
        if trainer.is_world_process_zero():
            handle_metrics(logger, "train", metrics, training_args.output_dir)
        trainer.save_state()

    # Evaluation
    max_length = (
        training_args.generation_max_length
        if training_args.generation_max_length is not None
        else data_args.val_max_answer_length
    )
    num_beams = data_args.num_beams if data_args.num_beams is not None else training_args.generation_num_beams

    if training_args.do_eval:
        if trainer.is_world_process_zero():
            # By default we load  the model from last checkpoint path,
            # in case of saving the model with the best metrics, make sure to
            # set save_total = 1 so the best model is loaded here.
            # if not exists returns the path to the output_dir.
            last_checkpoint_path = get_last_checkpoint(training_args.output_dir)
            model.load_state_dict(torch.load(os.path.join(last_checkpoint_path, "pytorch_model.bin"), map_location='cpu'))

            trainer = Seq2SeqHyperTrainer(
                train_dataset_sizes=train_dataset_sizes,
                val_dataset_sizes=eval_dataset_sizes,
                use_multitask_sampler=False,
                model=model,
                args=training_args,
                train_dataset=train_dataset if training_args.do_train else None,
                eval_dataset=eval_dataset if training_args.do_eval else None,
                tokenizer=tokenizer,
                data_collator=data_collator,
                compute_metrics=compute_metrics_fn if training_args.predict_with_generate else None,
            )

    if training_args.do_eval:
        logger.info("*** Evaluate ***")
        metrics = trainer.evaluate(max_length=max_length, num_beams=num_beams, metric_key_prefix="eval")

        if trainer.is_world_process_zero():
            handle_metrics(logger, "eval", metrics, training_args.output_dir)

    if training_args.do_test:
        trainer.eval_dataset = test_dataset
        trainer.val_dataset_sizes = test_dataset_sizes
        metrics = trainer.evaluate(test_dataset)
        if trainer.is_world_process_zero():
            handle_metrics(logger, "test", metrics, training_args.output_dir)


def _mp_fn(index):
    # For xla_spawn (TPUs)
    main()


if __name__ == "__main__":
    main()