using System.Linq;
using System.Collections.Generic;
using System.Diagnostics;

namespace SpreadsheetLLM.Heuristic
{
    internal partial class TableDetectionHybrid
    {
        private SheetMap _sheet;
        private List<Boundary> _regionGrowthBoxes;

        /// <summary>
        /// Candidate boxes field that is shared across all implementations of <see cref="TableDetectionHybrid"/> class.
        /// </summary>
        private List<Boundary> _boxes = new List<Boundary>();

        private IList<string> _logs;

        public TableDetectionHybrid(IList<string> logs)
        {
            _logs = logs;
        }

        public List<Boundary> Detect(SheetMap inputSheet, bool eliminateOverlaps)
        {
            var tableSenseDetectStopWatch = Stopwatch.StartNew();
            _sheet = inputSheet;
            //if file too big
            if (_sheet.Height > 1000 || _sheet.Height * _sheet.Width > 30000)
            {
                _logs.Add("Input too large - fall back to RegionGrowthDetect()");
                //simple region growth method
                RegionGrowthDetect();
                _logs.Add($"RegionGrowthDetect() ElapsedTime: {tableSenseDetectStopWatch.ElapsedMilliseconds}");
                tableSenseDetectStopWatch.Restart();
            }
            else
            {
                _logs.Add("Run TableSenseDetect()");
                TableSenseDetect();
                _logs.Add($"TableSenseDetect() ElapsedTime: {tableSenseDetectStopWatch.ElapsedMilliseconds}");
                tableSenseDetectStopWatch.Restart();

            }

            if (eliminateOverlaps)
            {
                _logs.Add($"Eliminating overlaps in {_boxes.Count} boxes");
                _boxes = Utils.RankBoxesBySize(_boxes);
                EliminateOverlaps();
                _logs.Add($"EliminateOverlaps. ElapsedTime: {tableSenseDetectStopWatch.ElapsedMilliseconds}");
                tableSenseDetectStopWatch.Restart();
            }

            _logs.Add($"Ranking {_boxes.Count} boxes");
            _boxes = Utils.RankBoxesByLocation(_boxes);

            _logs.Add($"Returning {_boxes.Count} boxes");
            tableSenseDetectStopWatch.Stop();
            return _boxes;
        }

        private void RegionGrowthDetect(int threshHor = 1, int threshVer = 1)
        {
            // main function for region growth method
            _boxes = RegionGrowthDetector.FindConnectedRanges(_sheet.contentStrs, _sheet.valueMapBorder, threshHor, threshVer);

            _logs.Add($"Found {_boxes.Count} connected ranges");

            // filter the little and sparse boxes
            LittleBoxesFilter();

            _logs.Add($"Filtered to {_boxes.Count} boxes");

            // filter the boxes overlaped the pivot tables
            // OverlapPivotFilter();

            // refine the upper boundary of the boxes to compact, especially when header exists
            UpHeaderTrim();

            // refine the boundaries of the boxes
            SurroundingBoudariesTrim();

            // extend the up header 
            RetrieveUpHeader(1);
            RetrieveUpHeader(2);

            // extend the left header 
            RetrieveLeftHeader();
            RetrieveLeft(1);
            RetrieveLeft(2);
        }

        private void TableSenseDetect()
        {
            // calculate the rowDiffsBool and colDiffsBool map and generate rowLine and colLine
            ////// extract the rowLine generation to a independent mothod
            _sheet.ProposeBoundaryLines();

            _logs.Add($"Proposed {_sheet.colBoundaryLines.Count} boundary lines");

            //  different types of cohension regions , often can not be partly overlaped by a correct box
            _sheet.CohensionDetection();

            // split the sheet to several discotinuous parts
            ////// TODO add a new parameter, the threshhold for the discotinuty
            List<Boundary> blockRegions = _sheet.GenerateBlockRegions();

            _logs.Add($"Generated {blockRegions.Count} block regions");

            List<Boundary> sheetBoxes = new List<Boundary>();
            List<Boundary> regiongrowthBoxes = new List<Boundary>();

            // generate and filter candidates for each blockRegion separately
            // int proposalsThresh = 4000;
            // int areaThresh = 10000;
            bool largeThresh = false;
            if (_sheet.Height * _sheet.Width > 10000)
            {
                largeThresh = true;
            }

            // use regions proposed by the region growth method
            int threshHorLimit = 7;
            for (int threshHor = 1; threshHor < threshHorLimit; threshHor++)
            {
                int threshVerLimit;
                if (threshHor < 3) threshVerLimit = 3;
                else threshVerLimit = 2;
                for (int threshVer = 1; threshVer < threshVerLimit; threshVer++)
                {
                    _boxes = RegionGrowthDetector.FindConnectedRanges(_sheet.contentStrs, _sheet.valueMapBorder, threshHor, threshVer, 0);

                    foreach (var box in _boxes)
                    {
                        if (GeneralFilter(box))
                        {
                            regiongrowthBoxes.Add(box);
                        }
                    }
                    if (!largeThresh)
                    {
                        _boxes = RegionGrowthDetector.FindConnectedRanges(_sheet.contentStrs, null, threshHor, threshVer, 0);
                        foreach (var box in _boxes)
                        {
                            if (GeneralFilter(box))
                            {
                                regiongrowthBoxes.Add(box);
                            }
                        }
                    }

                }
            }

            _logs.Add($"Found {regiongrowthBoxes.Count} region growth boxes");

            if (!largeThresh)
            {
                foreach (var blockRegion in blockRegions)
                {
                    _boxes = GenerateRawCandidateBoxes(blockRegion);
                    foreach (var box in regiongrowthBoxes)
                    {
                        if (Utils.isOverlap(blockRegion, box))
                        {
                            _boxes.Add(box);
                        }
                    }
                    // filter candidate boxes in blockregion
                    _boxes = Utils.DistinctBoxes(_boxes);
                    BlockCandidatesRefineAndFilter();

                    sheetBoxes.AddRange(_boxes);
                }
            }

            _logs.Add($"Found {sheetBoxes.Count} sheet boxes");

            _boxes = sheetBoxes;
            _boxes = Utils.DistinctBoxes(_boxes);
            _boxes = Utils.RankBoxesByLocation(_boxes);
            // filter candidates over all boxes in the sheet
            CandidatesRefineAndFilter();

            _logs.Add($"Filtered to {_boxes.Count} boxes");
        }

        private List<Boundary> GenerateRawCandidateBoxes(Boundary blockReigion)
        {
            List<Boundary> blockReigionboxes = new List<Boundary>();

            #region generate boundary lines in blockregion
            // find out the rowlines and collines in blockregion
            List<int> rowBoundaryLinesBlock = new List<int>();
            List<int> colBoundaryLinesBlock = new List<int>();

            foreach (int row in _sheet.rowBoundaryLines)
            {
                if (row < blockReigion.top - 2 || row > blockReigion.bottom - 1)
                {
                    continue;
                }
                for (int index = blockReigion.left - 3; index < blockReigion.right; index++)
                {
                    Boundary boxUp = new Boundary(row + 1, row + 1, index, index + 3);
                    Boundary boxDown = new Boundary(row + 2, row + 2, index, index + 3);

                    // exist inhomogeneity between the upside and downside
                    if (((_sheet.sumContentExist.SubmatrixSum(boxUp) > 0) && (_sheet.sumContentExist.SubmatrixSum(boxDown) == 0))
                        || ((_sheet.sumContentExist.SubmatrixSum(boxDown) > 0) && (_sheet.sumContentExist.SubmatrixSum(boxUp) == 0)))
                    {
                        rowBoundaryLinesBlock.Add(row);
                        break;
                    }
                }
            }

            rowBoundaryLinesBlock = rowBoundaryLinesBlock.Distinct().ToList();

            foreach (int col in _sheet.colBoundaryLines)
            {
                if (col < blockReigion.left - 2 || col > blockReigion.right - 1)
                {
                    continue;
                }

                for (int index = blockReigion.top - 3; index < blockReigion.bottom; index++)
                {
                    Boundary boxLeft = new Boundary(index, index + 3, col + 1, col + 1);
                    Boundary boxRight = new Boundary(index, index + 3, col + 2, col + 2);
                    // ununiform between the leftside and rightside
                    if (((_sheet.sumContentExist.SubmatrixSum(boxLeft) > 0) && (_sheet.sumContentExist.SubmatrixSum(boxRight) == 0))
                        || ((_sheet.sumContentExist.SubmatrixSum(boxRight) > 0) && (_sheet.sumContentExist.SubmatrixSum(boxLeft) == 0)))
                    {
                        colBoundaryLinesBlock.Add(col);
                        break;
                    }
                }
            }
            colBoundaryLinesBlock = colBoundaryLinesBlock.Distinct().ToList();
            #endregion

            // define complex blockregion regarding to the numbers of rowline and colline
            //TODO Threshhold index
            bool markComplex = false;
            if (rowBoundaryLinesBlock.Count > 300)
            {
                rowBoundaryLinesBlock.RemoveRange(300, rowBoundaryLinesBlock.Count - 300);
                markComplex = true;
            }
            if (colBoundaryLinesBlock.Count > 150)
            {
                colBoundaryLinesBlock.RemoveRange(150, colBoundaryLinesBlock.Count - 150);
                markComplex = true;
            }

            #region  generate candidates 
            // combinations of boundary lines as candidate regions
            foreach (int left in colBoundaryLinesBlock)
            {
                if (left < blockReigion.left - 2) continue;
                foreach (int right in colBoundaryLinesBlock)
                {
                    if (left >= right) continue;
                    if (right > blockReigion.right - 1) continue;
                    Dictionary<int, bool> notValidWorDown = new Dictionary<int, bool>();
                    foreach (int up in rowBoundaryLinesBlock)
                    {
                        if (up < blockReigion.top - 2) continue;
                        Boundary boxUpOut = new Boundary(up + 1, up + 1, left + 2, right + 1);
                        if (_sheet.sumContentExist.SubmatrixSum(boxUpOut) >= 6) continue;
                        Boundary boxLeftOut = new Boundary(up + 2, up + 4, left + 1, left + 1);
                        if (_sheet.sumContentExist.SubmatrixSum(boxLeftOut) >= 6) continue;
                        Boundary boxRightOut = new Boundary(up + 2, up + 4, right + 2, right + 2);
                        if (_sheet.sumContentExist.SubmatrixSum(boxRightOut) >= 6) continue;

                        Boundary boxUpIn = new Boundary(up + 2, up + 2, left + 2, right + 1);
                        if (_sheet.sumContentExist.SubmatrixSum(boxUpIn) == 0) continue;
                        foreach (int down in rowBoundaryLinesBlock)
                        {
                            if (up >= down) continue;
                            if (down > blockReigion.bottom - 1) continue;
                            if (notValidWorDown.ContainsKey(down)) continue;

                            Boundary boxDownOut = new Boundary(down + 2, down + 2, left + 2, right + 1);
                            Boundary boxDownIn = new Boundary(down + 1, down + 1, left + 2, right + 1);
                            if (_sheet.sumContentExist.SubmatrixSum(boxDownOut) >= 6) { notValidWorDown[down] = true; continue; }
                            if (_sheet.sumContentExist.SubmatrixSum(boxDownIn) == 0) { notValidWorDown[down] = true; continue; }
                            // generate box 
                            Boundary box = new Boundary(up + 2, down + 1, left + 2, right + 1);
                            // ensure the density
                            if (markComplex && _sheet.sumAll.SubmatrixSum(box) / Utils.AreaSize(box) < 2 * 0.3)
                            {
                                continue;
                            }

                            if (GeneralFilter(box))
                            {
                                blockReigionboxes.Add(box);
                            }
                        }
                    }
                }
            }
            #endregion
            return blockReigionboxes;
        }

        private void BlockCandidatesRefineAndFilter()
        {
            _logs.Add($"BlockCandidatesRefineAndFilter running on {_boxes.Count} boxes");

            // refine the upper boundary of the boxes to compact, especially when header exists
            UpHeaderTrim();

            // filter the boxes overlaped the sheet.forcedConhensionRegions
            OverlapCohensionFilter();
            // filter the boxes overlaped the sheet.forcedBorderRegions
            OverlapBorderCohensionFilter();
            // filter the boxes overlaped the pivot tables
            // overlapPivotFilter(ref boxes);

            _logs.Add($"{_boxes.Count} boxes after overlap filters");

            // filter little and sparse boxes
            LittleBoxesFilter();

            _logs.Add($"{_boxes.Count} boxes after filtering little boxes");

            // filter boxes that overlap the header incorrectly
            OverlapUpHeaderFilter();

            // refine the clarity of Boudary lines 
            SurroundingBoudariesTrim();

            // filter boxes that overlap the header incorrectly
            OverlapUpHeaderFilter();

            _logs.Add($"{_boxes.Count} boxes after header and boundary filters");

            // find out continuous empty rows/cols that can split the box into two irrelevant regions
            SplittedEmptyLinesFilter();

            // resolve suppression conficts of candidats boxes in soft manner
            ////SuppressionSoftFilter();
            // resolve suppression conficts of candidats boxes in hard manner
            ////SuppressionHardFilter();

            _logs.Add($"{_boxes.Count} boxes after empty lines and supression filters");

            //proProcessEnlarge();
        }

        private void CandidatesRefineAndFilter()
        {
            _logs.Add($"CandidatesRefineAndFilter running on {_boxes.Count} boxes");

            // add candidate boxes formed by borders
            BorderCohensionsAddition();

            _logs.Add($"{_boxes.Count} boxes after adding border cohensions");

            // filter little and sparse boxes
            LittleBoxesFilter();

            _logs.Add($"{_boxes.Count} boxes after filtering small boxes");

            // found out the up header apart from the data region
            RetrieveDistantUpHeader();

            // connect boxes with similar neighboring rows
            VerticalRelationalMerge();

            _logs.Add($"{_boxes.Count} boxes after relational merge");

            // resolve suppression conficts of candidats boxes
            SuppressionSoftFilter();

            _logs.Add($"{_boxes.Count} boxes after resolving suppression conflicts");

            // filter boxes missed header
            HeaderPriorityFilter();

            _logs.Add($"{_boxes.Count} boxes after filtering missed header");

            // combine boxes with formula reference
            // change
            ////FormulaCorrelationFilter();

            _logs.Add($"{_boxes.Count} boxes after formula reference combine");

            // solve the contradict containing  pairs with same similar left-right boundaries or similar up-down boundaries
            PairAlikeContainsFilter();
            // solve the contradict containing  pairs
            PairContainsFilter();

            _logs.Add($"{_boxes.Count} boxes after pair filters");

            // filter big boxes with compact  sub boxes containing header
            ////CombineContainsHeaderFilter();

            _logs.Add($"{_boxes.Count} boxes after contains header filter");

            // filter big  boxes with  sparse  sub boxes 
            CombineContainsFillAreaFilterSoft();
            // filter big  boxes with row-col direction  sub boxes containing header
            CombineContainsFillLineFilterSoft();
            // filter small  boxes with row-col direction  sub boxes containing header
            ContainsLittleFilter();
            // solve the contradict containing  pairs with same left-right or up-down

            _logs.Add($"{_boxes.Count} boxes after combine/contains filters");

            PairAlikeContainsFilter();
            // solve the contradict containing pairs
            PairContainsFilter();

            // in nesting combination cases, filter the intermediate candidates
            NestingCombinationFilter();

            OverlapHeaderFilter();

            _logs.Add($"{_boxes.Count} boxes after pair, nesting, and overlap filters");

            _boxes = Utils.DistinctBoxes(_boxes);

            _logs.Add($"{_boxes.Count} boxes after DistinctBoxes()");

            // preserve the boxes same as border regions and remove the other boxes overlap with them
            ForcedBorderFilter();

            // broad suppression filter 

            // two candidate boxes, with their headers overlapping each others
            AdjoinHeaderFilter();

            _logs.Add($"{_boxes.Count} boxes after border and adjoining filters");

            // filter little and sparse boxes
            LittleBoxesFilter();

            _logs.Add($"{_boxes.Count} boxes after small boxes filter");

            // hard filter 
            PairContainsFilterHard();
            CombineContainsFilterHard();

            _logs.Add($"{_boxes.Count} boxes after contains filters");

            // complete som region growth tables if missed
            AddRegionGrowth();
            AddCompactRegionGrowth();

            _logs.Add($"{_boxes.Count} boxes after region growth");

            MergeFilter();

            _logs.Add($"{_boxes.Count} boxes after merge");

            // if early retrieve header, then isheaderup will faled in containsFilter10
            // extend the up header 
            // extend the left  header 
            RetrieveLeftHeader();
            LeftHeaderTrim();
            BottomTrim();
            RetrieveUpHeader(1);
            RetrieveUpHeader(2);
            //retrieveUpHeader(ref boxes, 2);
            UpTrimSimple();
            //retrieveLeft(ref boxes, 1);
            //retrieveLeft(ref boxes, 2);

            //refineBoundaries(ref boxes);

            LittleBoxesFilter();

            _logs.Add($"{_boxes.Count} boxes concluding CandidatesRefineAndFilter");
        }

        #region merge Boxes
        private void VerticalRelationalMerge()
        {
            // connect boxes with similar neighboring rows
            var removedBoxes = new HashSet<Boundary>();
            var appendBoxes = new HashSet<Boundary>();

            List<Boundary> edgeforcedConhensionRegions = new List<Boundary>();
            // find out regions that could connect two boxes
            for (int i = 0; i < _boxes.Count; i++)
            {
                Boundary box = _boxes[i];

                Boundary boxDownRow = Utils.DownRow(box);

                Boundary boxDownRow2 = Utils.DownRow(boxDownRow, -1);

                // brittle part
                for (int k = 1; k < 6; k++)
                {
                    boxDownRow2 = Utils.DownRow(boxDownRow, -k);
                    // left header or none sparse contents
                    if ((_sheet.sumContentExist.SubmatrixSum(Utils.LeftCol(boxDownRow2)) != 0 && k > 1) || _sheet.sumContentExist.SubmatrixSum(boxDownRow2) >= 4) { break; }
                }
                if (_sheet.ContentExistValueDensity(boxDownRow2) >= 2 * 0.2 && _sheet.sumContentExist.SubmatrixSum(boxDownRow2) >= 4
                    && (_sheet.ComputeSimilarRow(boxDownRow, boxDownRow2) < 0.1)
                    && !IsHeaderUp(boxDownRow2) && !_sheet.ExistsMerged(boxDownRow2)
                    )
                {
                    Boundary forcedboxDown = new Boundary(boxDownRow.top, boxDownRow2.top, box.left, box.right);
                    edgeforcedConhensionRegions.Add(forcedboxDown);
                }

                Boundary boxUpRow = Utils.UpRow(box);

                Boundary boxUpRow2 = Utils.UpRow(boxUpRow, -1);

                // brittle part
                for (int k = 1; k < 6; k++)
                {
                    boxUpRow2 = Utils.UpRow(boxUpRow, -k);
                    // left header or none sparse contents
                    if ((_sheet.sumContentExist.SubmatrixSum(Utils.LeftCol(boxUpRow2)) != 0 && k > 1) || _sheet.sumContentExist.SubmatrixSum(boxUpRow2) >= 4) { break; }
                }
                if (_sheet.ContentExistValueDensity(boxUpRow2) >= 2 * 0.2 && _sheet.sumContentExist.SubmatrixSum(boxUpRow2) >= 4
                    && (_sheet.ComputeSimilarRow(boxUpRow, boxUpRow2) < 0.1)
                    && !IsHeaderUp(boxUpRow) && !_sheet.ExistsMerged(boxUpRow)
                    )
                {
                    Boundary forcedboxUp = new Boundary(boxUpRow2.top, boxUpRow.top, box.left, box.right);
                    edgeforcedConhensionRegions.Add(forcedboxUp);
                }
            }

            edgeforcedConhensionRegions = Utils.DistinctBoxes(edgeforcedConhensionRegions);

            foreach (var forcedbox in edgeforcedConhensionRegions)
            {
                List<Boundary> removedBoxesList = new List<Boundary>();
                // remove boxes overlap edgeforcedConhensionRegions
                foreach (var box in _boxes)
                {
                    if (forcedbox.left == box.left && forcedbox.right == box.right && Utils.isOverlap(box, forcedbox) && !Utils.ContainsBox(box, forcedbox))
                    {
                        removedBoxes.Add(box);
                        if (box.top < forcedbox.top || box.bottom > forcedbox.bottom)
                        {
                            removedBoxesList.Add(box);
                        }
                    }
                }
                // add nex combined boxes from the removed ones
                //foreach (var box1 in removedBoxesList)
                //{
                //    foreach (var box2 in removedBoxesList)
                //    {
                //        Boundary newBox = new Boundary { Math.Min(box1.up, Math.Min(forcedbox.up, box2.up)), Math.Max(box1.down, Math.Max(forcedbox.down, box2.down)), box1.left, box1.right };
                //        appendBoxes.Add(newBox);
                //    }
                //}
            }

            Utils.RemoveAndAppendCandidates(removedBoxes, appendBoxes, _boxes);
        }
        #endregion

        #region forced connected cohension or border cohension
        private List<Boundary> BorderCohensionFilterSuppression(List<Boundary> borderCohensions)
        {
            var removedBoxes = new HashSet<Boundary>();
            foreach (var box1 in _boxes)
            {
                foreach (var borderCohension in borderCohensions)
                {
                    if (Utils.isSuppressionBox(box1, borderCohension, step: 2) && !box1.Equals(borderCohension))
                    {
                        removedBoxes.Add(borderCohension);
                    }
                }
            }
            //foreach (var box in removedBoxes.Keys)
            //    boxes.Remove(box);
            foreach (var box in removedBoxes)
                borderCohensions.Remove(box);
            return borderCohensions;
        }

        private void BorderCohensionsAddition()
        {
            // add candidate boxes formed by borders
            List<Boundary> borderRegions = new List<Boundary>();
            foreach (var box in _sheet.cohensionBorderRegions)
            {
                borderRegions.Add(box);
            }
            foreach (var box in _sheet.smallCohensionBorderRegions)
            {
                if (box.bottom - box.top > 1 && box.right - box.left > 1)
                {
                    borderRegions.Add(box);
                }

            }

            List<Boundary> clearBorderRegions = new List<Boundary>();
            foreach (var box in borderRegions)
            {
                Boundary boxUp = Utils.UpRow(box, -1);
                Boundary boxDown = Utils.DownRow(box, -1);
                Boundary boxLeft = Utils.LeftCol(box, -1);
                Boundary boxRight = Utils.RightCol(box, -1);
                if (box.right - box.left > 3 && _sheet.ContentExistValueDensity(boxUp) >= 2 * 0.5)
                {
                    continue;
                }
                if (box.right - box.left > 3 && _sheet.ContentExistValueDensity(boxDown) >= 2 * 0.5)
                {
                    continue;
                }
                if (box.bottom - box.top > 3 && _sheet.ContentExistValueDensity(boxLeft) >= 2 * 0.5)
                {
                    continue;
                }
                if (box.bottom - box.top > 3 && _sheet.ContentExistValueDensity(boxRight) >= 2 * 0.5)
                {
                    continue;
                }

                clearBorderRegions.Add(box);
            }
            // filter some border boxes 
            List<Boundary> candidateRanges = BorderCohensionFilterSuppression(clearBorderRegions);

            _boxes.AddRange(candidateRanges);
            _boxes = Utils.DistinctBoxes(_boxes);
        }
        #endregion
        
        #region region growth
        private void AddRegionGrowth(int threshHor = 1, int threshVer = 1)
        {
            // generate candidate boxes predicted by the region growth method
            _regionGrowthBoxes = RegionGrowthDetector.FindConnectedRanges(_sheet.contentStrs, null, threshHor, threshVer);

            // append the candidates that proposed by addRegionGrowth method but not overlap with anyone in the current boxes
            var appendBoxes = new HashSet<Boundary>();
            var removedBoxes = new HashSet<Boundary>();

            foreach (var box in _regionGrowthBoxes)
            {
                if (_sheet.sumContentExist.SubmatrixSum(box) < 24) continue;
                List<Boundary> boxesIn = new List<Boundary>();
                List<Boundary> boxesOverlap = new List<Boundary>();
                foreach (var box2 in _boxes)
                {
                    if (Utils.isOverlap(box2, box))
                    {
                        boxesOverlap.Add(box2);
                    }
                    if (Utils.ContainsBox(box, box2, 1) && !Utils.isSuppressionBox(box, box2))
                    {
                        boxesIn.Add(box2);
                    }
                }
                // if there are no overlaps, then append this box
                if (boxesOverlap.Count == 0)
                {
                    appendBoxes.Add(box);
                }
                // if  if there is only one overlap, then decide which one to preserve by comparing the relative densities
                else if (boxesOverlap.Count == 1 && boxesIn.Count == 1 && Utils.AreaSize(box) - Utils.AreaSize(boxesIn[0]) > 20
                    && (_sheet.sumContentExist.SubmatrixSum(box) - _sheet.sumContentExist.SubmatrixSum(boxesIn[0])) / (Utils.AreaSize(box) - Utils.AreaSize(boxesIn[0])) > 0.5 * 2)
                {
                    appendBoxes.Add(box);
                    removedBoxes.Add(box);
                }
            }
            Utils.RemoveAndAppendCandidates(removedBoxes, appendBoxes, _boxes);
            _boxes = Utils.DistinctBoxes(_boxes);
        }

        private void AddCompactRegionGrowth()
        {
            // generate compact candidate boxes predicted by the region growth method
            var regionGrowthCompactBoxes = ProProcessReduceToCompact(_regionGrowthBoxes);
            regionGrowthCompactBoxes = ProProcessReduceToCompact(regionGrowthCompactBoxes);

            // append the candidates proposed that not overlap with anyone in the current boxes
            var appendBoxes = new HashSet<Boundary>();
            foreach (var box in regionGrowthCompactBoxes)
            {
                if (_sheet.sumContentExist.SubmatrixSum(box) < 7) continue;
                if (_sheet.sumContentExist.SubmatrixSum(new Boundary(box.top - 1, box.bottom + 1, box.left - 1, box.right + 1)) - _sheet.sumContentExist.SubmatrixSum(box) > 10) continue;

                if (!Utils.isOverlap(box, _boxes))
                {
                    appendBoxes.Add(box);
                }
            }
            Utils.AppendTheseCandidates(appendBoxes, _boxes);
        }
        #endregion
    }
}
