using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using SheetCore;

namespace SpreadsheetLLM.Heuristic
{
    public static class HeuristicTableDetector
    {
        private const int WindowHeight = 5;
        private const int WindowWidth = 5;
        /// <summary>
        /// The heuristic TableSense performance is relative to input grid size (Height*Width), but not total non-blank cells in the grid.
        /// In extreme cases, a chunk with 10000 cells may shape an 10000*10000 window that causes significant overhead.
        /// Therefore, we introduce <see cref="MaximumInputGridSize"/> that defines an upperbound of input grid size.
        /// </summary>
        private const int MaximumInputGridSize = 250000;
        private static readonly List<Boundary> EmptyIntArrayList = new List<Boundary>();

        public static IList<IRange> Detect(ISheet inputSheet, bool eliminateOverlaps, IList<string> logs)
        {
            var result = DetectBoundary(inputSheet, eliminateOverlaps, logs);
            return result.Select(box => (IRange)new Range(box.top, box.bottom, box.left, box.right)).ToList();
        }

        public static IList<IRange> HybridCombine(ISheet inputSheet, IList<IRange> heuResults, IList<IRange> mlResults)
        {
            int height = inputSheet.Height;
            int width = inputSheet.Width;
            if (height == 0 && width == 0)
            {
                return System.Array.Empty<IRange>();
            }
            if (height * width > MaximumInputGridSize)
            {
                return System.Array.Empty<IRange>();
            }

            // Construct a Sheet object from inputSheet
            var sheet = new CoreSheet(inputSheet);

            CellFeatures.ExtractFeatures(sheet, out var features, out var cells, out var formula);
            var sheetMap = new SheetMap(WindowHeight + 2, WindowWidth + 2, features, cells, formula, sheet.MergedAreas, EmptyIntArrayList);

            var logs = new List<string>();
            var detector = new TableDetectionHybrid(new List<string>());

            var heuBoundaries = heuResults.Select(boundary => new Boundary(
                up: boundary.FirstRow - inputSheet.FirstRow + WindowHeight + 3,
                down: boundary.LastRow - inputSheet.FirstRow + WindowHeight + 3,
                left: boundary.FirstColumn - inputSheet.FirstColumn + WindowWidth + 3,
                right: boundary.LastColumn - inputSheet.FirstColumn + WindowWidth + 3
            )).ToList();

            var mlBoundaries = mlResults.Select(boundary => new Boundary(
                up: boundary.FirstRow - inputSheet.FirstRow + WindowHeight + 3,
                down: boundary.LastRow - inputSheet.FirstRow + WindowHeight + 3,
                left: boundary.FirstColumn - inputSheet.FirstColumn + WindowWidth + 3,
                right: boundary.LastColumn - inputSheet.FirstColumn + WindowWidth + 3
            )).ToList();

            var boxes = detector.HybridCombine(sheetMap, heuBoundaries, mlBoundaries);

            return boxes.Select(box => (IRange)new Range(
                box.top + inputSheet.FirstRow - WindowHeight - 3, // start_row, 1 indexed
                box.bottom + inputSheet.FirstRow - WindowHeight - 3, // end_row
                box.left + inputSheet.FirstColumn - WindowWidth - 3, // start_col
                box.right + inputSheet.FirstColumn - WindowWidth - 3 // end_col
            )).ToArray();
        }

        internal static List<Boundary> DetectBoundary(SheetCore.ISheet inputSheet, bool eliminateOverlaps, IList<string> logs)
        {
            int height = inputSheet.Height;
            int width = inputSheet.Width;
            if (height == 0 && width == 0)
            {
                logs.Add("Zero sized input sheet");
                return EmptyIntArrayList;
            }
            if (height * width > MaximumInputGridSize)
            {
                logs.Add("Skipped large sized input sheet");
                return EmptyIntArrayList;
            }

            // Construct a Sheet object from inputSheet
            var sheet = new CoreSheet(inputSheet);

            Stopwatch stopwatch = Stopwatch.StartNew();
            CellFeatures.ExtractFeatures(sheet, out var features, out var cells, out var formula);
            logs.Add($"ExtractFeatures ElapsedTime: {stopwatch.ElapsedMilliseconds}");
            stopwatch.Restart();
            var sheetMap = new SheetMap(WindowHeight + 2, WindowWidth + 2, features, cells, formula, sheet.MergedAreas, EmptyIntArrayList);
            logs.Add($"SheetMap: {stopwatch.ElapsedMilliseconds}");
            stopwatch.Restart();
            var detector = new TableDetectionHybrid(logs);
            var boxes = detector.Detect(sheetMap, eliminateOverlaps);  // 1-indexed, based on left-top cell
            logs.Add($"TableDetectionHybrid.Detect. ElapsedTime: {stopwatch.ElapsedMilliseconds}");
            for (int i = 0; i < boxes.Count; i++)
            {
                var box = boxes[i];
                box.top += inputSheet.FirstRow - WindowHeight - 3;     // start_row, 1 indexed
                box.bottom += inputSheet.FirstRow - WindowHeight - 3;  // end_row
                box.left += inputSheet.FirstColumn - WindowWidth - 3;  // start_col
                box.right += inputSheet.FirstColumn - WindowWidth - 3; // end_col
                boxes[i] = box;
            }

            return boxes;
        }
    }
}
