from __future__ import annotations

import math
import re
import subprocess
import json

from pydantic import BaseModel

from hedal.keypack import KeyPack
from hedal.load_heaan import HedalParameter


class Context:
    def __init__(self, parameter: HedalParameter, make_bootstrappable: bool = True, use_gpu: bool = True):
        self.parameter = parameter
        self.make_bootstrappable = make_bootstrappable

        self.context = self.heaan.make_context(self.parameter.params)
        self.log_slots = self.heaan.get_log_full_slots(self.context)
        
        if make_bootstrappable and self.context.is_bootstrappable_parameter:
            self.heaan.make_bootstrappable(self.context)

        self.__secret_key = None
        self.__public_key = None

        self.__encryptor = self.heaan.Encryptor(self.context)
        self.__decryptor = self.heaan.Decryptor(self.context)
        self.__homevaluator = None

        self.__with_gpu = False
        if use_gpu:
            self.with_gpu = True

    @property
    def public_key(self):
        if not self.__public_key:
            raise Exception("Require load_pk")
        return self.__public_key

    @property
    def secret_key(self):
        if not self.__secret_key:
            raise Exception("require load_sk")
        return self.__secret_key
    
    @property
    def heaan(self):
        return self.parameter.heaan

    @property
    def heaan_type(self):
        return self.parameter.heaan_type

    @property
    def num_slots(self):
        return 1 << self.log_slots

    @property
    def shape(self):
        if math.log2(self.num_slots) % 2 == 0:
            return (int(math.sqrt(self.num_slots)), int(math.sqrt(self.num_slots)))
        else:
            return (int(math.sqrt(self.num_slots/2)), int(math.sqrt(self.num_slots*2)))

    @property
    def min_level_for_bootstrap(self):
        return self.__homevaluator.min_level_for_bootstrap

    @property
    def encryptor(self):
        return self.__encryptor

    @property
    def decryptor(self):
        return self.__decryptor

    @property
    def homevaluator(self):
        if not self.__homevaluator:
            raise Exception("Do after generate_homevaluator")
        return self.__homevaluator

    def load_pk(self, path):
        if not self.__public_key:
            self.__public_key = KeyPack.load_public_key(self, path)

    def load_sk(self, path):
        if not self.__secret_key:
            self.__secret_key = KeyPack.load_secret_key(self, path)

    def generate_homevaluator(self):
        self.__homevaluator = self.heaan.HomEvaluator(self.context, self.public_key)

    def history(self, verbose: bool = False):
        if self.heaan_type == "pi":
            return self.context.repr_history(verbose)
        else:
            raise TypeError(f"Unsupported method for {self.heaan_type}")

    @property
    def with_gpu(self) -> bool:
        return self.__with_gpu

    @with_gpu.setter
    def with_gpu(self, status: bool) -> None:
        if status:
            try:
                b_out = subprocess.check_output(["nvcc", "--version"])
                out = b_out.decode("utf-8")
                nvcc_version = float(re.findall(r"release ([\d\.]*)", out)[0])
                if nvcc_version > 11.2:
                    self.__with_gpu = True
                    print(f"HEDAL use CUDA v{nvcc_version:.1f} (> v11.2)")
                else:
                    self.__with_gpu = False
                    print(f"CUDA v{nvcc_version:.1f} <= v11.2 is not supported")
            except:
                self.__with_gpu = False
                print("CUDA is not available.")
        else:
            self.__with_gpu = False
            print(f"HEDAL do not use CUDA.")

    @property
    def metadata(self):
        return self.Metadata(
            heaan_type=self.heaan_type,
            id=self.parameter.id,
            make_bootstrappable=self.make_bootstrappable
        )

    class Metadata(BaseModel):
        heaan_type: str
        id: str
        make_bootstrappable: bool

    @staticmethod
    def load(path: str) -> Context:
        with open(path, "r") as m_file:
            m_info = json.load(m_file)
        parameter = HedalParameter.from_preset(id=m_info["id"], heaan_type=m_info["heaan_type"])
        return Context(parameter=parameter, make_bootstrappable=m_info["make_bootstrappable"]) 

    def save(self, path: str) -> None:
        with open(path, "w") as m_file:
            m_file.write(self.metadata.json())