# from transformers.data.datasets.squad import *
# from transformers.data.processors.squad import *
# from multiprocessing import Pool, cpu_count

import json
import os
from functools import partial
from multiprocessing import Pool, cpu_count

import numpy as np
from tqdm import tqdm

from transformers.file_utils import is_torch_available
from transformers.tokenization_bert import whitespace_tokenize
from transformers.tokenization_utils_base import TruncationStrategy
from transformers.utils import logging
from transformers.data.processors.utils import DataProcessor
from data.dataset_utils import get_prefix_tokens

logger = logging.get_logger(__name__)

MULTI_SEP_TOKENS_TOKENIZERS_SET = {"roberta", "camembert", "bart"}

if is_torch_available():
    import torch
    from torch.utils.data import TensorDataset

def _improve_answer_span(doc_tokens, input_start, input_end, tokenizer, orig_answer_text):
    """Returns tokenized answer spans that better match the annotated answer."""

    candidate_cleaned_forms = []
    candidate_cleaned_forms.append(" ".join(tokenizer.tokenize(orig_answer_text)))
    if tokenizer.__class__.__name__ in [
            "RobertaTokenizer",
            "BartTokenizer",
        ]:
        candidate_cleaned_forms.append(" ".join(tokenizer.tokenize(orig_answer_text, add_prefix_space=True)))

    for new_start in range(input_start, input_end + 1):
        for new_end in range(input_end, new_start - 1, -1):
            text_span = " ".join(doc_tokens[new_start : (new_end + 1)])
            if text_span in candidate_cleaned_forms:
                return (new_start, new_end)

    return (input_start, input_end)


def _check_is_max_context(doc_spans, cur_span_index, position):
    """Check if this is the 'max context' doc span for the token."""
    best_score = None
    best_span_index = None
    for (span_index, doc_span) in enumerate(doc_spans):
        end = doc_span.start + doc_span.length - 1
        if position < doc_span.start:
            continue
        if position > end:
            continue
        num_left_context = position - doc_span.start
        num_right_context = end - position
        score = min(num_left_context, num_right_context) + 0.01 * doc_span.length
        if best_score is None or score > best_score:
            best_score = score
            best_span_index = span_index

    return cur_span_index == best_span_index


def _new_check_is_max_context(doc_spans, cur_span_index, position):
    """Check if this is the 'max context' doc span for the token."""
    # if len(doc_spans) == 1:
    # return True
    best_score = None
    best_span_index = None
    for (span_index, doc_span) in enumerate(doc_spans):
        end = doc_span["start"] + doc_span["length"] - 1
        if position < doc_span["start"]:
            continue
        if position > end:
            continue
        num_left_context = position - doc_span["start"]
        num_right_context = end - position
        score = min(num_left_context, num_right_context) + 0.01 * doc_span["length"]
        if best_score is None or score > best_score:
            best_score = score
            best_span_index = span_index

    return cur_span_index == best_span_index


def _is_whitespace(c):
    if c == " " or c == "\t" or c == "\r" or c == "\n" or ord(c) == 0x202F:
        return True
    return False

def custom_squad_convert_example_to_features_init(tokenizer_for_convert):
    global tokenizer
    tokenizer = tokenizer_for_convert

def custom_squad_convert_examples_to_features(
    examples,
    tokenizer,
    max_seq_length,
    max_query_length,
    is_training,
    padding_strategy="max_length",
    return_dataset=False,
    threads=1,
    tqdm_enabled=True,
    dataset='hpqa,'
):
    """
    Converts a list of examples into a list of features that can be directly given as input to a model.
    It is model-dependant and takes advantage of many of the tokenizer's features to create the model's inputs.

    Args:
        examples: list of :class:`~transformers.data.processors.squad.SquadExample`
        tokenizer: an instance of a child of :class:`~transformers.PreTrainedTokenizer`
        max_seq_length: The maximum sequence length of the inputs.
        max_query_length: The maximum length of the query.
        is_training: whether to create features for model evaluation or model training.
        padding_strategy: Default to "max_length". Which padding strategy to use
        return_dataset: Default False. Either 'pt' or 'tf'.
            if 'pt': returns a torch.data.TensorDataset,
            if 'tf': returns a tf.data.Dataset
        threads: multiple processing threadsa-smi


    Returns:
        list of :class:`~transformers.data.processors.squad.SquadFeatures`

    Example::

        processor = SquadV2Processor()
        examples = processor.get_dev_examples(data_dir)

        features = squad_convert_examples_to_features(
            examples=examples,
            tokenizer=tokenizer,
            max_seq_length=args.max_seq_length,
            max_query_length=args.max_query_length,
            is_training=not evaluate,
        )
    """

    # Defining helper methods
    features = []
    threads = min(threads, cpu_count())
    with Pool(threads, initializer=custom_squad_convert_example_to_features_init, initargs=(tokenizer,)) as p:
        annotate_ = partial(
            custom_squad_convert_example_to_features,
            max_seq_length=max_seq_length,
            max_query_length=max_query_length,
            padding_strategy=padding_strategy,
            is_training=is_training,
            dataset=dataset,
        )
        features = list(
            tqdm(
                p.imap(annotate_, examples, chunksize=32),
                total=len(examples),
                desc="convert squad examples to features",
                disable=not tqdm_enabled,
            )
        )
    new_features = []
    unique_id = 1000000000
    example_index = 0
    for example_features in tqdm(
        features, total=len(features), desc="add example index and unique id", disable=not tqdm_enabled
    ):
        if not example_features:
            continue
        for example_feature in example_features:
            example_feature.example_index = example_index
            example_feature.unique_id = unique_id
            new_features.append(example_feature)
            unique_id += 1
        example_index += 1
    features = new_features
    del new_features
    if return_dataset == "pt":
        if not is_torch_available():
            raise RuntimeError("PyTorch must be installed to return a PyTorch dataset.")

        # Convert to Tensors and build dataset
        all_input_ids = torch.tensor([f.input_ids for f in features], dtype=torch.long)
        all_attention_masks = torch.tensor([f.attention_mask for f in features], dtype=torch.long)
        all_token_type_ids = torch.tensor([f.token_type_ids for f in features], dtype=torch.long)
        all_cls_index = torch.tensor([f.cls_index for f in features], dtype=torch.long)
        all_p_mask = torch.tensor([f.p_mask for f in features], dtype=torch.float)
        all_is_impossible = torch.tensor([f.is_impossible for f in features], dtype=torch.float)

        if not is_training:
            all_feature_index = torch.arange(all_input_ids.size(0), dtype=torch.long)
            dataset = TensorDataset(
                all_input_ids, all_attention_masks, all_token_type_ids, all_feature_index, all_cls_index, all_p_mask
            )
        else:
            all_start_positions = torch.tensor([f.start_position for f in features], dtype=torch.long)
            all_end_positions = torch.tensor([f.end_position for f in features], dtype=torch.long)
            dataset = TensorDataset(
                all_input_ids,
                all_attention_masks,
                all_token_type_ids,
                all_start_positions,
                all_end_positions,
                all_cls_index,
                all_p_mask,
                all_is_impossible,
            )

        return features, dataset
    else:
        return features

def custom_squad_convert_example_to_features(
    example, max_seq_length, max_query_length, padding_strategy, is_training, dataset
):

    tok_to_orig_index = []
    orig_to_tok_index = []
    all_doc_tokens = []

    prefix_tokens = get_prefix_tokens(dataset, tokenizer)

    augmented_doc_tokens = prefix_tokens + example.doc_tokens

    ex_start_position = example.start_position + len(prefix_tokens)
    ex_end_position = example.end_position + len(prefix_tokens)
    # transform context by prepending athings
    # insert tokens for special purpose
    

    for (i, token) in enumerate(augmented_doc_tokens):
        orig_to_tok_index.append(len(all_doc_tokens))

        if i < len(prefix_tokens):
            tok_to_orig_index.append(i)
            all_doc_tokens.append(token)

        else:
            if tokenizer.__class__.__name__ in [
                "RobertaTokenizer",
                "BartTokenizer",
            ]:
                sub_tokens = tokenizer.tokenize(token, add_prefix_space=True)
            else:
                sub_tokens = tokenizer.tokenize(token)

            for sub_token in sub_tokens:
                tok_to_orig_index.append(i)
                all_doc_tokens.append(sub_token)

    if is_training:
        if example.is_yesno:
            tok_start_position = prefix_tokens.index(example.answer_text)
            tok_end_position = tok_start_position
        elif not example.is_impossible:
            tok_start_position = orig_to_tok_index[ex_start_position]
            if ex_end_position < len(augmented_doc_tokens) - 1:
                tok_end_position = orig_to_tok_index[ex_end_position + 1] - 1
            else:
                tok_end_position = len(all_doc_tokens) - 1

            (tok_start_position, tok_end_position) = _improve_answer_span(
                all_doc_tokens, tok_start_position, tok_end_position, tokenizer, example.answer_text
            )

    truncated_query = tokenizer.encode(
        example.question_text, add_special_tokens=False, truncation=True, max_length=max_query_length
    )

    # Tokenizers who insert 2 SEP tokens in-between <context> & <question> need to have special handling
    # in the way they compute mask of added tokens.
    tokenizer_type = type(tokenizer).__name__.replace("Tokenizer", "").lower()
    sequence_added_tokens = (
        tokenizer.model_max_length - tokenizer.max_len_single_sentence + 1
        if tokenizer_type in MULTI_SEP_TOKENS_TOKENIZERS_SET
        else tokenizer.model_max_length - tokenizer.max_len_single_sentence
    )
    sequence_pair_added_tokens = tokenizer.model_max_length - tokenizer.max_len_sentences_pair

    span_doc_tokens = all_doc_tokens
    
    # Define the side we want to truncate / pad and the text/pair sorting
    if tokenizer.padding_side == "right":
        texts = truncated_query
        pairs = span_doc_tokens
        truncation = TruncationStrategy.ONLY_SECOND.value
    else:
        texts = span_doc_tokens
        pairs = truncated_query
        truncation = TruncationStrategy.ONLY_FIRST.value
    encoded_dict = tokenizer.encode_plus(  # TODO(thom) update this logic
        texts,
        pairs,
        truncation=truncation,
        padding=padding_strategy,
        max_length=max_seq_length,
        return_overflowing_tokens=False,
        return_token_type_ids=True,
    )

    paragraph_len = min(
        len(all_doc_tokens),
        max_seq_length - len(truncated_query) - sequence_pair_added_tokens,
    )

    if tokenizer.pad_token_id in encoded_dict["input_ids"]:
        if tokenizer.padding_side == "right":
            non_padded_ids = encoded_dict["input_ids"][: encoded_dict["input_ids"].index(tokenizer.pad_token_id)]
        else:
            last_padding_id_position = (
                len(encoded_dict["input_ids"]) - 1 - encoded_dict["input_ids"][::-1].index(tokenizer.pad_token_id)
            )
            non_padded_ids = encoded_dict["input_ids"][last_padding_id_position + 1 :]

    else:
        non_padded_ids = encoded_dict["input_ids"]

    tokens = tokenizer.convert_ids_to_tokens(non_padded_ids)

    token_to_orig_map = {}
    for i in range(paragraph_len):
        index = len(truncated_query) + sequence_added_tokens + i if tokenizer.padding_side == "right" else i
        token_to_orig_map[index] = tok_to_orig_index[i]

    encoded_dict["paragraph_len"] = paragraph_len
    encoded_dict["tokens"] = tokens
    encoded_dict["token_to_orig_map"] = token_to_orig_map
    encoded_dict["truncated_query_with_special_tokens_length"] = len(truncated_query) + sequence_added_tokens
    encoded_dict["token_is_max_context"] = {}
    encoded_dict["start"] = 0
    encoded_dict["length"] = paragraph_len

    span = encoded_dict

    # while end

    doc_span_index = 0
    for j in range(span["paragraph_len"]):
        is_max_context = _new_check_is_max_context([span], doc_span_index, j)
        index = (
            j
            if tokenizer.padding_side == "left"
            else span["truncated_query_with_special_tokens_length"] + j
        )
        span["token_is_max_context"][index] = is_max_context

    # for span in spans:
    # Identify the position of the CLS token
    cls_index = span["input_ids"].index(tokenizer.cls_token_id)

    # p_mask: mask with 1 for token than cannot be in the answer (0 for token which can be in an answer)
    # Original TF implem also keep the classification token (set to 0)
    p_mask = np.ones_like(span["token_type_ids"])
    if tokenizer.padding_side == "right":
        p_mask[len(truncated_query) + sequence_added_tokens :] = 0
    else:
        p_mask[-len(span["tokens"]) : -(len(truncated_query) + sequence_added_tokens)] = 0

    pad_token_indices = np.where(span["input_ids"] == tokenizer.pad_token_id)
    special_token_indices = np.asarray(
        tokenizer.get_special_tokens_mask(span["input_ids"], already_has_special_tokens=True)
    ).nonzero()

    p_mask[pad_token_indices] = 1
    p_mask[special_token_indices] = 1

    # Set the cls index to 0: the CLS index can be used for impossible answers
    p_mask[cls_index] = 0

    span_is_impossible = example.is_impossible
    start_position = 0
    end_position = 0
    if is_training and not span_is_impossible:
        # For training, if our document chunk does not contain an annotation
        # we throw it out, since there is nothing to predict.
        doc_start = span["start"]
        doc_end = span["start"] + span["length"] - 1
        out_of_span = False

        if not (tok_start_position >= doc_start and tok_end_position <= doc_end):
            out_of_span = True

        if out_of_span:
            start_position = cls_index
            end_position = cls_index
            span_is_impossible = True
        else:
            if tokenizer.padding_side == "left":
                doc_offset = 0
            else:
                doc_offset = len(truncated_query) + sequence_added_tokens

            start_position = tok_start_position - doc_start + doc_offset
            end_position = tok_end_position - doc_start + doc_offset
    features = []
    features.append(
        SquadFeatures(
            span["input_ids"],
            span["attention_mask"],
            span["token_type_ids"],
            cls_index,
            p_mask.tolist(),
            example_index=0,  # Can not set unique_id and example_index here. They will be set after multiple processing.
            unique_id=0,
            paragraph_len=span["paragraph_len"],
            token_is_max_context=span["token_is_max_context"],
            tokens=span["tokens"],
            token_to_orig_map=span["token_to_orig_map"],
            start_position=start_position,
            end_position=end_position,
            is_impossible=span_is_impossible,
            qas_id=example.qas_id,
        )
    )
    return features


class SquadProcessor(DataProcessor):
    """
    Processor for the SQuAD data set.
    Overriden by SquadV1Processor and SquadV2Processor, used by the version 1.1 and version 2.0 of SQuAD, respectively.
    """


    def get_train_examples(self, data_dir, filename=None):
        """
        Returns the training examples from the data directory.
        Args:
            data_dir: Directory containing the data files used for training and evaluating.
            filename: None by default, specify this if the training file has a different name than the original one
                which is `train-v1.1.json` and `train-v2.0.json` for squad versions 1.1 and 2.0 respectively.
        """
        if data_dir is None:
            data_dir = ""

        with open(
            os.path.join(data_dir, filename), "r", encoding="utf-8"
        ) as reader:
            input_data = json.load(reader)["data"]
        return self._create_examples(input_data, "train")

    def get_dev_examples(self, data_dir, filename=None):
        """
        Returns the evaluation example from the data directory.
        Args:
            data_dir: Directory containing the data files used for training and evaluating.
            filename: None by default, specify this if the evaluation file has a different name than the original one
                which is `train-v1.1.json` and `train-v2.0.json` for squad versions 1.1 and 2.0 respectively.
        """
        if data_dir is None:
            data_dir = ""

        with open(
            os.path.join(data_dir, filename), "r", encoding="utf-8"
        ) as reader:
            input_data = json.load(reader)["data"]
        return self._create_examples(input_data, "dev")

    def _create_examples(self, input_data, set_type, tqdm_enabled=True):
        is_training = set_type == "train"
        examples = []
        for entry in tqdm(input_data, disable=not tqdm_enabled):
            title = entry["title"]
            for paragraph in entry["paragraphs"]:
                context_text = paragraph["context"]

                for qa in paragraph["qas"]:
                    qas_id = qa["id"]
                    question_text = qa["question"]
                    start_position_character = None
                    answer_text = None
                    answers = []
                    question_type = qa.get('question_type','none')
                    is_yesno = qa.get('is_yesno', False)
                    is_impossible = qa.get("is_impossible", False)
                    if not is_impossible:
                        if is_training:
                            answer = qa["answers"][0]
                            answer_text = answer["text"]
                            start_position_character = answer["answer_start"]
                        else:
                            answers = qa["answers"]
                    example = SquadExample(
                        qas_id=qas_id,
                        question_text=question_text,
                        context_text=context_text,
                        answer_text=answer_text,
                        start_position_character=start_position_character,
                        title=title,
                        is_yesno=is_yesno,
                        is_impossible=is_impossible,
                        question_type=question_type,
                        answers=answers,
                    )
                    examples.append(example)
        return examples


class SquadExample:
    """
    A single training/test example for the Squad dataset, as loaded from disk.
    Args:
        qas_id: The example's unique identifier
        question_text: The question string
        context_text: The context string
        answer_text: The answer string
        start_position_character: The character position of the start of the answer
        title: The title of the example
        answers: None by default, this is used during evaluation. Holds answers as well as their start positions.
        is_impossible: False by default, set to True if the example has no possible answer.
    """

    def __init__(
        self,
        qas_id,
        question_text,
        context_text,
        answer_text,
        start_position_character,
        title,
        answers=[],
        question_type=None,
        is_yesno=False,
        is_impossible=False,
    ):
        self.qas_id = qas_id
        self.question_text = question_text
        self.context_text = context_text
        self.answer_text = answer_text
        self.title = title
        self.is_impossible = is_impossible
        self.is_yesno = is_yesno
        self.question_type = question_type
        self.answers = answers

        self.start_position, self.end_position = 0, 0

        doc_tokens = []
        char_to_word_offset = []
        prev_is_whitespace = True

        # Split on whitespace so that different tokens may be attributed to their original position.
        for c in self.context_text:
            if _is_whitespace(c):
                prev_is_whitespace = True
            else:
                if prev_is_whitespace:
                    doc_tokens.append(c)
                else:
                    doc_tokens[-1] += c
                prev_is_whitespace = False
            char_to_word_offset.append(len(doc_tokens) - 1)

        self.doc_tokens = doc_tokens
        self.char_to_word_offset = char_to_word_offset

        # Start and end positions only has a value during evaluation.
        if self.is_yesno:
            self.start_position = -1
            self.end_position = -1
        else:
            if start_position_character is not None and not is_impossible:
                self.start_position = char_to_word_offset[start_position_character]
                self.end_position = char_to_word_offset[
                    min(start_position_character + len(answer_text) - 1, len(char_to_word_offset) - 1)
                ]

class SquadFeatures:
    """
    Single squad example features to be fed to a model.
    Those features are model-specific and can be crafted from :class:`~transformers.data.processors.squad.SquadExample`
    using the :method:`~transformers.data.processors.squad.squad_convert_examples_to_features` method.
    Args:
        input_ids: Indices of input sequence tokens in the vocabulary.
        attention_mask: Mask to avoid performing attention on padding token indices.
        token_type_ids: Segment token indices to indicate first and second portions of the inputs.
        cls_index: the index of the CLS token.
        p_mask: Mask identifying tokens that can be answers vs. tokens that cannot.
            Mask with 1 for tokens than cannot be in the answer and 0 for token that can be in an answer
        example_index: the index of the example
        unique_id: The unique Feature identifier
        paragraph_len: The length of the context
        token_is_max_context: List of booleans identifying which tokens have their maximum context in this feature object.
            If a token does not have their maximum context in this feature object, it means that another feature object
            has more information related to that token and should be prioritized over this feature for that token.
        tokens: list of tokens corresponding to the input ids
        token_to_orig_map: mapping between the tokens and the original text, needed in order to identify the answer.
        start_position: start of the answer token index
        end_position: end of the answer token index
    """

    def __init__(
        self,
        input_ids,
        attention_mask,
        token_type_ids,
        cls_index,
        p_mask,
        example_index,
        unique_id,
        paragraph_len,
        token_is_max_context,
        tokens,
        token_to_orig_map,
        start_position,
        end_position,
        is_impossible,
        qas_id: str = None,
    ):
        self.input_ids = input_ids
        self.attention_mask = attention_mask
        self.token_type_ids = token_type_ids
        self.cls_index = cls_index
        self.p_mask = p_mask

        self.example_index = example_index
        self.unique_id = unique_id
        self.paragraph_len = paragraph_len
        self.token_is_max_context = token_is_max_context
        self.tokens = tokens
        self.token_to_orig_map = token_to_orig_map

        self.start_position = start_position
        self.end_position = end_position
        self.is_impossible = is_impossible
        self.qas_id = qas_id


class SquadResult:
    """
    Constructs a SquadResult which can be used to evaluate a model's output on the SQuAD dataset.
    Args:
        unique_id: The unique identifier corresponding to that example.
        start_logits: The logits corresponding to the start of the answer
        end_logits: The logits corresponding to the end of the answer
    """

    def __init__(self, unique_id, start_logits, end_logits, start_top_index=None, end_top_index=None, cls_logits=None):
        self.start_logits = start_logits
        self.end_logits = end_logits
        self.unique_id = unique_id

        if start_top_index:
            self.start_top_index = start_top_index
            self.end_top_index = end_top_index
            self.cls_logits = cls_logits
