import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
from typing import Any, Dict
from transformers import BartModel, BartForConditionalGeneration

from torchfly.training import FlyModel
from torchfly.metrics import CategoricalAccuracy, Average, MovingAverage, Speed


class BartGenerationFlyModel(FlyModel):
    def __init__(self, config):
        super().__init__(config)
        self.bart = BartForConditionalGeneration.from_pretrained(config.task.pretrained_model, dropout=0.0, attention_dropout=0.0, activation_dropout=0.0)
        # configure metrics here
        self.configure_metrics()

    def configure_metrics(self):
        self.training_metrics = {"loss": MovingAverage()}
        self.evaluation_metrics = {"loss": Average()}

    def forward(self, batch: Dict[str, Any]) -> Dict[str, Any]:
        batch = batch[0]
        output = self.bart(input_ids=batch["encoder_input_ids"], 
                            decoder_input_ids=batch["decoder_input_ids"],
                           attention_mask=batch["encoder_attention_mask"], 
                           labels=batch["target"], 
                           return_dict=True)
        self.training_metrics["loss"](output.loss.item())
        return output

    def predict_step(self, batch):
        output = self.bart(input_ids=batch["encoder_input_ids"], 
                           decoder_input_ids=batch["decoder_input_ids"],
                           attention_mask=batch["encoder_attention_mask"], 
                           labels=batch["target"], return_dict=True)
        self.evaluation_metrics["loss"](output.loss.item())
        return None

    def get_training_metrics(self) -> Dict[str, str]:
        loss = self.training_metrics["loss"].get_metric()
        metrics = {"loss": f"{loss:.4f}"}
        return metrics

    def get_evaluation_metrics(self) -> Dict[str, str]:
        loss = self.evaluation_metrics["loss"].get_metric()
        ppl = np.exp(loss)
        metrics = {"loss": (f"{loss:.4f}", loss), "ppl": (
            f"{ppl:.4f}", ppl), "score": (f"{-ppl:.4f}", -ppl)}
        return metrics
