package fractaldragon.topicanalysis;

import de.ruedigermoeller.serialization.FSTObjectInput;
import de.ruedigermoeller.serialization.FSTObjectOutput;
import fractaldragon.SharedConstants;
import fractaldragon.util.IdKeyCountsMap;
import fractaldragon.util.Tree;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Iterator;
import java.util.ListIterator;

/**
 * Class to save and retrieve model state for the hierarchical model
 * for use in subsequent processing steps as reporting or summarization.
 */
public class HierarchicalModelStoreXtra implements Serializable {
    int maxTopic;
    int numTypes;
    int maxLevel;
    Tree<Integer> tø;
    int[] n;
    int[] m;
    IdKeyCountsMap idKeyCounts;

    void allocateStructures(int size) {
        // Create empty allocated structures when building from scratch.
        // Used by flat to hierarchical conversion.
        tø = new Tree<>(null, 0);
        n = new int[size];
        m = new int[size];
       // idKeyCounts added separately.
    }
    // public getters for outside of package.
    public int maxTopic() { return maxTopic; }
    public int numTypes() { return numTypes; }
    public int maxLevel() { return maxLevel; }
    public Tree<Integer> tø() { return tø; }
    public int[] n() { return n; }
    public int[] m() { return m; }
    public IdKeyCountsMap idKeyCounts() { return idKeyCounts; }
    
    // For inside of package we can set directly.
    // But the mGE structure may not be present in the HDP structure so we 
    // can construct it before saving the model. This must be invoked after
    // setting tø and m, but before saving the model.
    public void buildMGE() {
        assert m != null;
        assert tø != null;       
    }
    
    
    /**
     * Service methods accessible within the package.
     * @param tree
     * @param table 
     */
    static public void initializeReverseT(Tree<Integer> tree, Tree<Integer>[] table) {
        // Tree assumed to be non-null, so it has an iterator.
        Arrays.fill(table, null);
        Iterator<Tree<Integer>> iter = tree.iterator();
        while (iter.hasNext()) {
            Tree<Integer> node = iter.next();
            table[node.getValue()] = node;
        }    
    }

    @Override
    public String toString() {
        return maxTopic+", "+numTypes+", "+maxLevel+"\n"+Arrays.toString(n)+
          "\n"+Arrays.toString(m)+"\n"+idKeyCounts+"\n"+tø.toTreeString();
    }
    
    public static HierarchicalModelStoreXtra readModel(String modelFN) 
      throws FileNotFoundException, IOException, ClassNotFoundException, Exception {
        File file = new File(modelFN+SharedConstants.serSfx.suffix());
        try(FileInputStream stream = new FileInputStream (file)) {
            switch (SharedConstants.serSfx) {
                case SER:
                    try (ObjectInputStream in = new ObjectInputStream(stream)) {
                        return (HierarchicalModelStoreXtra)in.readObject();       
                    }                    
                case FST:
                    try (FSTObjectInput in = new FSTObjectInput(stream)) {
                        return (HierarchicalModelStoreXtra)in.readObject(HierarchicalModelStoreXtra.class);       
                    }
            }
            return null;
        }
    }
    
    public static void writeModel(HierarchicalModelStoreXtra model, String modelFN) 
      throws FileNotFoundException, IOException {
        File file = new File(modelFN+SharedConstants.serSfx.suffix());
        try(FileOutputStream stream = new FileOutputStream(file)) { 
            switch (SharedConstants.serSfx) {
                case SER:
                    try (ObjectOutputStream out = new ObjectOutputStream(stream)) {
                        out.writeObject(model);
                        break;  
                    }                           
                case FST:
                    try (FSTObjectOutput out = new FSTObjectOutput(stream)) {
                        //System.out.println("Model:"+model);
                        out.writeObject(model, HierarchicalModelStoreXtra.class );
                        break;  
                    }       
            }
        }
    }
    
    public void writeModel(String modelFN) throws IOException {
        writeModel(this, modelFN);
    }
}
