feature_descriptions = dict()


class FeaturesMixin:
    descriptions = dict()

    def __init__(self):
        self.features = 0

    @staticmethod
    def feature_info(bit: int, desc: str):
        def decorator(func):
            feature_descriptions[getattr(func, "__name__")] = (bit, desc)
            return func
        return decorator

    def is_bit_enabled(self, offset: int) -> bool:
        return (self.features & (1 << offset)) != 0

    @property
    @feature_info(0, "Use an hidden state produced by an earlier layer in the Transformer for NER")
    def is_feature_offset_ner_hidden_state(self):
        return self.features & (1 << 0) != 0

    @property
    @feature_info(1, "Sum representations")
    def is_feature_sum_representations(self):
        return self.features & (1 << 1) != 0

    @property
    @feature_info(2, "  + Attention")
    def is_feature_attend_representations(self):
        return self.features & (1 << 2) != 0

    @property
    @feature_info(3, "  + Attention for all positions")
    def is_feature_attend_all_positions(self):
        return self.features & (1 << 3) != 0

    @property
    @feature_info(4, "  + KV from frozen encoder")
    def is_feature_attend_kv_from_frozen_encoder(self):
        return self.features & (1 << 4) != 0

    @property
    @feature_info(5, "Extra LR class")
    def is_feature_extra_lr_class(self):
        return self.features & (1 << 5) != 0

    @property
    @feature_info(6, "Inference v2 (removed)")
    def is_feature_inference_v2(self):
        return self.features & (1 << 6) != 0

    @property
    @feature_info(7, "Extra RR class")
    def is_feature_extra_rr_class(self):
        return self.features & (1 << 7) != 0

    @property
    @feature_info(8, "Single link type")
    def is_feature_single_link_type(self):
        # unused in the code but used in the individual datasets
        return self.features & (1 << 8) != 0

    @property
    @feature_info(9, "Nest depth > 1")
    def is_feature_nest_depth_gt_1(self):
        return self.features & (1 << 9) != 0

    @property
    @feature_info(10, "Optimized inference")
    def is_feature_perf_optimized(self):
        return self.features & (1 << 10) != 0

    @property
    @feature_info(11, "NER Only")
    def is_feature_ner_only(self):
        return self.features & (1 << 11) != 0

    @property
    @feature_info(12, "Frozen encoder")
    def is_feature_frozen_encoder(self):
        return self.features & (1 << 12) != 0

    @property
    @feature_info(13, "Empty examples")
    def is_feature_empty_examples(self):
        return self.features & (1 << 13) != 0

    def list_features(self):
        for k, v in FeaturesMixin.__dict__.items():
            if not k.startswith("is_feature_"):
                continue
            active = getattr(self, k)
            bit, desc = feature_descriptions[k]
            if active:
                print(f"Using feature '{desc}' (bit {bit}, {1 << bit})")
