using System;
using System.Linq;
using System.Text.RegularExpressions;
using System.Collections.Generic;

namespace SpreadsheetLLM.Heuristic
{
    internal class SheetMap
    {
        public int Height;
        public int Width;

        int rowOffset;
        int colOffset;

        public List<Boundary> pivotBoxes;
        public List<Boundary> mergeBoxes;

        public string[,] contentStrs;

        public string[,] formuStrs;
        public List<Boundary>[,] formulaRanges;

        public CellFeatures[,] featureMap;

        public int[,] valueMapContent;
        public int[,] valueMapBorder;
        public int[,] valueMapColor;
        public int[,] valueMapAll;

        public int[,] sumContent;
        public int[,] sumContentExist;
        public int[,] sumBorder;
        public int[,] sumBorderCol;
        public int[,] sumBorderRow;
        public int[,] sumColor;
        public int[,] sumAll;

        public List<int> rowBoundaryLines;
        public List<int> colBoundaryLines;

        List<List<bool>> rowDiffsBool;
        List<List<bool>> colDiffsBool;
        List<List<double>> rowDiffs;
        List<List<double>> colDiffs;


        public List<Boundary> conhensionRegions;
        public List<Boundary> mergeforcedConhensionRegions;
        public List<Boundary> colforcedConhensionRegions;
        public List<Boundary> rowforcedConhensionRegions;
        public List<Boundary> colorforcedConhensionRegions;
        public List<Boundary> edgeforcedConhensionRegions;
        public List<Boundary> cohensionBorderRegions;
        public List<Boundary> smallCohensionBorderRegions;

        public SheetMap(int rowOffsetNum, int colOffsetNum, CellFeatures[,] features, string[,] contents, string[,] formus,
            List<Boundary> mergedRanges, List<Boundary> pivotTables)
        {
            // the row and column distance to shift the sheet 
            rowOffset = rowOffsetNum;
            colOffset = colOffsetNum;

            // list of pivottables
            pivotBoxes = ExtendPivot(pivotTables);

            // merged Regions in the sheet
            mergeBoxes = ConvertMergedRanges(mergedRanges);

            // shift the strings by rowOffset and colOffset in the sheet
            featureMap = ExtendFeatureMap(features, rowOffset, colOffset, out Height, out Width);
            contentStrs = ExtendMatrix(contents, rowOffset, colOffset);
            formuStrs = ExtendMatrix(formus, rowOffset, colOffset);

            // find the formula reference regions by regular expressions match
            formulaRanges = preProcessFormulas(formuStrs);

            // update the feature map for the sheet by the formulas 
            UpdateFeatAndContentsByFormula();

            // calculate the many types of value maps
            CalculateBasicValueMaps();
        }

        private void CalculateBasicValueMaps()
        {
            // maps
            valueMapContent = ComputeValueMap(feature => (feature.HasFormula ? 2 : 0) + (feature.MarkText ? 2 : 0));
            int[,] valueMapContentExist = ComputeValueMap(feature => feature.HasFormula || feature.MarkText ? 2 : 0);
            valueMapBorder = ComputeValueMap(feature =>
            {
                int cntBorder = Convert.ToInt32(feature.HasBottomBorder) + Convert.ToInt32(feature.HasTopBorder) + Convert.ToInt32(feature.HasLeftBorder) + Convert.ToInt32(feature.HasRightBorder);
                return (cntBorder >= 3 ? cntBorder - 1 : cntBorder) * 2;
            });
            int[,] valueMapBorderCol = ComputeValueMap(feature => feature.HasLeftBorder || feature.HasRightBorder ? 2 : 0);
            int[,] valueMapBorderRow = ComputeValueMap(feature => feature.HasBottomBorder || feature.HasTopBorder ? 2 : 0);
            valueMapColor = ComputeValueMap(feature => feature.HasFillColor ? 2 : 0);
            valueMapAll = ComputeValueMap((i, j) => Math.Min(valueMapContent[i, j] + valueMapBorder[i, j] + valueMapColor[i, j], 16));

            // sum maps
            sumContent = valueMapContent.CalcSumMatrix();
            sumContentExist = valueMapContentExist.CalcSumMatrix();
            sumBorder = valueMapBorder.CalcSumMatrix();
            sumBorderCol = valueMapBorderCol.CalcSumMatrix();
            sumBorderRow = valueMapBorderRow.CalcSumMatrix();
            sumColor = valueMapColor.CalcSumMatrix();
            sumAll = valueMapAll.CalcSumMatrix();
        }

        public void CohensionDetection()
        {
            // detect different types of cohension regions , often can not be partly overlaped by a correct box
            colforcedConhensionRegions = new List<Boundary>();
            rowforcedConhensionRegions = new List<Boundary>();
            colorforcedConhensionRegions = new List<Boundary>();
            mergeforcedConhensionRegions = new List<Boundary>();
            edgeforcedConhensionRegions = new List<Boundary>();

            // several kinds of forced compact regions that can not be cut off by a box
            conhensionRegions = new List<Boundary>();

            //  compact regions that formed by cell borders
            cohensionBorderRegions = new List<Boundary>();
            //  small compact regions that formed by cell borders
            smallCohensionBorderRegions = new List<Boundary>();


            //// generate several kinds of cohesion regions
            mergeforcedConhensionRegions = GenerateMergedCohensions(ref rowBoundaryLines, ref colBoundaryLines);
            colorforcedConhensionRegions = generateColorCohensions();
            //colforcedConhensionRegions = generateColContentCohensions(patienceThresh: 2, stepThresh: 3);
            //rowforcedConhensionRegions = generateRowContentCohensions(patienceThresh: 1, stepThresh: 1);

            // generate forcedBorderRegions and forcedBorderRegionsSmall
            dealWithBorderCohensionRegions();

            conhensionRegions.AddRange(colforcedConhensionRegions);
            conhensionRegions.AddRange(rowforcedConhensionRegions);

            conhensionRegions.AddRange(mergeforcedConhensionRegions);
        }

        #region Propose Boundary Lines
        public void ProposeBoundaryLines()
        {
            // initialise candidate boundary lines
            rowBoundaryLines = new List<int>();
            colBoundaryLines = new List<int>();

            //// compute valueMap differences
            computeValueDiff();

            //// propose box lines candidates in the row and column directions
            rowBoundaryLines = ProposeLinesByDiff(rowDiffsBool, lineDiffThresh: 1.0);
            colBoundaryLines = ProposeLinesByDiff(colDiffsBool, lineDiffThresh: 1.0);

            //// remove duplicates
            rowBoundaryLines = rowBoundaryLines.Distinct().ToList();
            colBoundaryLines = colBoundaryLines.Distinct().ToList();

            //// sort lines by the ununiform degree
            rowBoundaryLines.Sort((x, y) => -rowDiffsBool[x].Count(diff => diff).CompareTo(rowDiffsBool[y].Count(diff => diff)));
            colBoundaryLines.Sort((x, y) => -colDiffsBool[x].Count(diff => diff).CompareTo(colDiffsBool[y].Count(diff => diff)));
        }

        private static List<int> ProposeLinesByDiff(List<List<bool>> diffs, double lineDiffThresh)
        {
            return Enumerable.Range(0, diffs.Count)
                .Where(index => diffs[index].Count(diff => diff) >= lineDiffThresh)
                .ToList();
        }

        public double ComputeBorderDiffsRow(Boundary box)
        {
            int up = box.top - 2;
            int down = box.bottom - 1;
            int left = box.left - 2;
            int right = box.right - 1;

            double cnt1;
            double cnt2;

            cnt1 = 0;
            for (int j = left; j <= right; j++)
            {
                if (rowDiffsBool[up][j] & rowDiffs[up][j] > 0)
                    cnt1 = cnt1 + Math.Abs(rowDiffs[up][j]);
            }

            cnt2 = 0;
            for (int j = left; j <= right; j++)
            {
                if (rowDiffsBool[down][j] && rowDiffs[down][j] < 0)
                    cnt2 = cnt2 + Math.Abs(rowDiffs[down][j]);
            }

            double result = (cnt1 + cnt2) / Utils.Width(box);
            return result;
        }
        public double ComputeBorderDiffsCol(Boundary box)
        {
            int up = box.top - 2;
            int down = box.bottom - 1;
            int left = box.left - 2;
            int right = box.right - 1;

            double cnt3;
            double cnt4;

            cnt3 = 0;
            for (int i = up; i <= down; i++)
            {
                if (colDiffsBool[left][i] && colDiffs[left][i] > 0)
                    cnt3 = cnt3 + Math.Abs(colDiffs[left][i]);
            }

            cnt4 = 0;
            for (int i = up; i <= down; i++)
            {
                if (colDiffsBool[right][i] && colDiffs[right][i] < 0)
                    cnt4 = cnt4 + Math.Abs(colDiffs[right][i]);
            }

            double result = (cnt3 + cnt4) / Utils.Height(box);
            return result;
        }
        #endregion

        #region extend
        public List<Boundary> ExtendPivot(List<Boundary> pivotTables)
        {
            return pivotTables.Select(pivotBox => new Boundary(
                pivotBox.top + rowOffset,
                pivotBox.bottom + rowOffset,
                pivotBox.left + colOffset,
                pivotBox.right + colOffset)).ToList();
        }

        private static CellFeatures[,] ExtendFeatureMap(CellFeatures[,] features, int rowOffset, int colOffset, out int height, out int width)
        {
            height = features.GetLength(0) + 2 * rowOffset;
            width = features.GetLength(1) + 2 * colOffset;
            var results = new CellFeatures[height, width];

            for (int i = 0; i < height; i++)
            {
                for (int j = 0; j < width; j++)
                {
                    if (rowOffset <= i && i < height - rowOffset && colOffset <= j && j < width - colOffset)
                    {
                        results[i, j] = features[i - rowOffset, j - colOffset];
                    }
                    else
                    {
                        results[i, j] = CellFeatures.EmptyFeatureVec;
                    }
                }
            }

            return results;
        }

        private static string[,] ExtendMatrix(string[,] matrixIn, int offsetRow, int offsetCol)
        {
            string[,] matrixOut = new string[matrixIn.GetLength(0) + 2 * offsetRow, matrixIn.GetLength(1) + 2 * offsetCol];
            for (int i = 0; i < matrixOut.GetLength(0); i++)
            {
                for (int j = 0; j < matrixOut.GetLength(1); j++)
                {
                    if (offsetRow <= i && i < matrixOut.GetLength(0) - offsetRow && offsetCol <= j && j < matrixOut.GetLength(1) - offsetCol)
                    {
                        matrixOut[i, j] = matrixIn[i - offsetRow, j - offsetCol];
                    }
                    else
                    {
                        matrixOut[i, j] = "";
                    }
                }
            }
            return matrixOut;
        }
        #endregion

        #region block regions generation
        public List<Boundary> GenerateBlockRegions()
        {
            Boundary initialRegion = new Boundary(1, Height, 1, Width);
            List<Boundary> blockRegions = new List<Boundary>();
            List<Boundary> prevRegions = new List<Boundary> { initialRegion };
            List<Boundary> newRegions = new List<Boundary> { initialRegion };

            int cntSplit = 1;
            //iterate until no split generated
            while (cntSplit > 0)
            {
                prevRegions = newRegions;
                newRegions = new List<Boundary>();
                cntSplit = 0;
                for (int i = 0; i < prevRegions.Count; i++)
                {
                    if (prevRegions[i].bottom - prevRegions[i].top <= 0 || prevRegions[i].right - prevRegions[i].left <= 0)
                    {
                        continue;
                    }
                    // remove the  redundant borders
                    Boundary newbox = TrimEmptyEdges(prevRegions[i]);
                    if (!prevRegions[i].Equals(newbox))
                    {
                        newRegions.Add(newbox);
                        cntSplit++;
                        continue;
                    }
                    //split the region
                    List<Boundary> splitBoxes = SplitBlockRegion(prevRegions[i]);
                    if (splitBoxes.Count == 2)
                    {
                        newRegions.Add(splitBoxes[0]);
                        newRegions.Add(splitBoxes[1]);
                        cntSplit++;
                        continue;
                    }
                    blockRegions.Add(prevRegions[i]);
                }
            }
            return blockRegions;
        }

        /// <summary>
        /// Remove none edges from the up, down, left, right directions
        /// </summary>
        private Boundary TrimEmptyEdges(Boundary box)
        {
            int up = box.top;
            int down = box.bottom;
            int left = box.left;
            int right = box.right;

            for (; up < down; ++up)
            {
                Boundary edgeBox = new Boundary(up, up, left, right);
                if (sumContent.SubmatrixSum(edgeBox) + sumColor.SubmatrixSum(edgeBox) + sumBorder.SubmatrixSum(edgeBox) != 0)
                    break;
            }
            for (; down >= up; --down)
            {
                Boundary edgeBox = new Boundary(down, down, left, right);
                if (sumContent.SubmatrixSum(edgeBox) + sumColor.SubmatrixSum(edgeBox) + sumBorder.SubmatrixSum(edgeBox) != 0)
                    break;
            }
            for (; left < right; ++left)
            {
                Boundary edgeBox = new Boundary(up, down, left, left);
                if (sumContent.SubmatrixSum(edgeBox) + sumColor.SubmatrixSum(edgeBox) + sumBorder.SubmatrixSum(edgeBox) != 0)
                    break;
            }
            for (; right >= left; --right)
            {
                Boundary edgeBox = new Boundary(up, down, right, right);
                if (sumContent.SubmatrixSum(edgeBox) + sumColor.SubmatrixSum(edgeBox) + sumBorder.SubmatrixSum(edgeBox) != 0)
                    break;
            }

            return new Boundary(up, down, left, right);
        }

        private List<Boundary> SplitBlockRegion(Boundary box)
        {
            int up = box.top;
            int down = box.bottom;
            int left = box.left;
            int right = box.right;

            // horizontal split
            for (int i = up + 4; i <= down - 4; i++)
            {
                // one row without format and  three continuous rows without contents
                Boundary edgeBox5 = new Boundary(i - 2, i + 2, left, right);
                Boundary edgeBox3 = new Boundary(i - 1, i + 1, left, right);
                Boundary edgeBox1 = new Boundary(i, i, left, right);

                if (sumContent.SubmatrixSum(edgeBox1) + sumColor.SubmatrixSum(edgeBox1) == 0
                    && (sumContentExist.SubmatrixSum(edgeBox3) == 0
                    || (sumContentExist.SubmatrixSum(edgeBox3) < 6 && sumContentExist.SubmatrixSum(edgeBox5) < 10))
                    || (ContentExistValueDensity(edgeBox5) < 0.1 && sumContentExist.SubmatrixSum(edgeBox5) < 10))
                {
                    //find out the first unnone line below
                    int k = i + 2;
                    Boundary edgeBoxDown = new Boundary(k, k, left, right);
                    while (k < down)
                    {
                        edgeBoxDown = new Boundary(k, k, left, right);
                        if (sumContentExist.SubmatrixSum(edgeBoxDown) > 2) break;
                        k++;
                    }
                    //find out the first unnone line above
                    k = i - 2;
                    Boundary edgeBoxUp = new Boundary(k, k, left, right);
                    while (k > up)
                    {
                        edgeBoxUp = new Boundary(k, k, left, right);
                        if (sumContentExist.SubmatrixSum(edgeBoxUp) > 2) break;
                        k--;
                    }
                    // if not strictly related, then split
                    if (down - edgeBoxDown.top > 1 && edgeBoxUp.top - up > 1)
                    {
                        return new List<Boundary> { new Boundary(up, i - 1, left, right), new Boundary(i + 1, down, left, right) };
                    }
                }
            }
            // vertical split
            for (int i = left + 4; i <= right - 4; i++)
            {
                Boundary edgeBox1 = new Boundary(up, down, i, i);
                Boundary edgeBox3 = new Boundary(up, down, i - 1, i + 1);
                Boundary edgeBox5 = new Boundary(up, down, i - 2, i + 2);

                if (sumContent.SubmatrixSum(edgeBox1) + sumColor.SubmatrixSum(edgeBox1) == 0
                    && (sumContentExist.SubmatrixSum(edgeBox3) == 0
                    || (sumContentExist.SubmatrixSum(edgeBox3) < 6 && sumContentExist.SubmatrixSum(edgeBox5) < 10))
                    || (ContentExistValueDensity(edgeBox5) < 0.1 && sumContentExist.SubmatrixSum(edgeBox5) < 10))
                {
                    int k = i + 2;
                    Boundary edgeBoxRight = new Boundary(up, down, k, k);
                    while (k <= right)
                    {
                        edgeBoxRight = new Boundary(up, down, k, k);
                        if (sumContentExist.SubmatrixSum(edgeBoxRight) > 2) break;
                        k++;
                    }

                    k = i - 2;
                    Boundary edgeBoxLeft = new Boundary(up, down, k, k);
                    while (k >= left)
                    {
                        edgeBoxLeft = new Boundary(up, down, k, k);
                        if (sumContentExist.SubmatrixSum(edgeBoxLeft) > 2) break;
                        k--;
                    }

                    if (right - edgeBoxRight.left > 1 && edgeBoxLeft.left - left > 1 && !verifyMutualMatchCol(edgeBoxRight, edgeBoxLeft))
                    {
                        return new List<Boundary> { new Boundary(up, down, left, i - 1), new Boundary(up, down, i + 1, right) };
                    }

                }
            }
            return new List<Boundary>();
        }
        #endregion

        #region  border cohension regions
        private void locateBorderCohensions(bool[,,] borderMap)
        {
            //search the regions filled with borders
            // to record which cells in the map has been already searched
            bool[,] borderMapMark = new bool[Height, Width];
            for (int i = 0; i < Height; i++)
            {
                for (int j = 0; j < Width; j++)
                {
                    borderMapMark[i, j] = true;
                }
            }

            for (int i = 0; i < Height; i++)
            {
                for (int j = 0; j < Width; j++)
                {
                    if (!borderMapMark[i, j]) continue;
                    int stepRow = 0;
                    int stepCol = 0;
                    //greedy search
                    while (i + stepRow < Height && j + stepCol < Width)
                    {
                        if (!borderMap[i + stepRow, j + stepCol, 0]) break;
                        if (!borderMap[i + stepRow, j + stepCol, 1]) break;
                        if (!borderMap[i + stepRow, j + stepCol, 2]) break;
                        if (!borderMap[i + stepRow, j + stepCol, 3]) break;
                        stepRow++;
                        stepCol++;
                    }
                    if (stepRow > 0) stepRow--;
                    if (stepCol > 0) stepCol--;
                    while (i + stepRow < Height)
                    {
                        if (!borderMap[i + stepRow, j + stepCol, 0]) break;
                        if (!borderMap[i + stepRow, j + stepCol, 1]) break;
                        if (!borderMap[i + stepRow, j + stepCol, 2]) break;
                        if (!borderMap[i + stepRow, j + stepCol, 3]) break;
                        stepRow++;
                    }
                    if (stepRow > 0) stepRow--;
                    while (j + stepCol < Width)
                    {
                        if (!borderMap[i + stepRow, j + stepCol, 0]) break;
                        if (!borderMap[i + stepRow, j + stepCol, 1]) break;
                        if (!borderMap[i + stepRow, j + stepCol, 2]) break;
                        if (!borderMap[i + stepRow, j + stepCol, 3]) break;
                        stepCol++;
                    }
                    if (stepCol > 0) stepCol--;

                    if (stepRow == 0 && stepCol == 0) continue;

                    // get the region
                    Boundary forcedBorderRegion = new Boundary(i + 1, i + 1 + stepRow, j + 1, j + 1 + stepCol);

                    // record this region in the borderMapMark
                    for (int p = 0; p <= stepRow; p++)
                    {
                        for (int q = 0; q <= stepCol; q++)
                        {
                            borderMapMark[i + p, j + q] = false;
                        }
                    }

                    if (CntNoneBorder(forcedBorderRegion) >= 2)
                    {
                        continue;
                    }

                    // verify the region, and add it to forcedBorderRegions
                    if (VerifyBorderRegion(forcedBorderRegion))
                    {
                        if (stepRow >= 3 && stepCol >= 2)
                        {
                            cohensionBorderRegions.Add(forcedBorderRegion);
                        }
                        else if (stepRow >= 2 || stepCol >= 2)
                        {
                            smallCohensionBorderRegions.Add(forcedBorderRegion);
                        }
                    }
                }
            }
        }

        private bool VerifyBorderRegion(Boundary borderRegion)
        {
            int up = borderRegion.top - 1;
            int down = borderRegion.bottom - 1;
            int left = borderRegion.left - 1;
            int right = borderRegion.right - 1;

            // +2 avoid header
            for (int col = left + 2; col <= right; col++)
            {////featuremap bottom in, same bottom border form left to right
                if (col > left && featureMap[down, col].HasBottomBorder != featureMap[down, col - 1].HasBottomBorder)
                {
                    return false;
                }
            }

            for (int row = up + 2; row <= down; row++)
            {////featuremap right in, need have same right border form up to down
                if (row > up && featureMap[row, right].HasRightBorder != featureMap[row - 1, right].HasRightBorder)
                {
                    return false;
                }
            }

            for (int col = left; col <= right; col++)
            {////featuremap bottom out, dont have left border
                if (col > left && featureMap[down + 1, col].HasLeftBorder)
                {
                    return false;
                }
            }

            for (int col = left; col <= right; col++)
            {////featuremap up out, dont have left border
                if (col > left && featureMap[up - 1, col].HasLeftBorder)
                {
                    return false;
                }
            }

            for (int row = up; row <= down; row++)
            {////featuremap right out, dont have up border
                if (row > up && featureMap[row, right + 1].HasTopBorder)
                {
                    return false;
                }
            }

            for (int row = up; row <= down; row++)
            {////featuremap left out, dont have up border
                if (row > up && featureMap[row, left - 1].HasTopBorder)
                {
                    return false;
                }
            }

            return true;
        }
        #endregion

        #region deal with border,merge,color,semantic row/col
        private void dealWithBorderCohensionRegions()
        {
            // preprocess the border maps
            // the subscript indexes, 0: bottom border, 1: top border, 2: left border, 3: right border
            bool[,,] borderMap = new bool[Height, Width, 4];

            // initialize and preprocess the border map
            for (int i = 0; i < Height; i++)
            {
                for (int j = 0; j < Width; j++)
                {
                    borderMap[i, j, 0] = featureMap[i, j].HasBottomBorder;
                    borderMap[i, j, 1] = featureMap[i, j].HasTopBorder;
                    borderMap[i, j, 2] = featureMap[i, j].HasLeftBorder;
                    borderMap[i, j, 3] = featureMap[i, j].HasRightBorder;

                    // the left border for the right cell is same as the right border for the left cell
                    if (borderMap[i, j, 0] && i - 1 >= 0)
                    {
                        borderMap[i - 1, j, 1] = true;
                    }
                    if (borderMap[i, j, 1] && i + 1 < Height)
                    {
                        borderMap[i + 1, j, 0] = true;
                    }
                    if (borderMap[i, j, 2] && j - 1 >= 0)
                    {
                        borderMap[i, j - 1, 3] = true;
                    }
                    if (borderMap[i, j, 3] && j + 1 < Width)
                    {
                        borderMap[i, j + 1, 2] = true;
                    }
                }
            }

            // in order to search for border regions conveniently, first fill the border map in some degree
            //for example, if a cell have up and right borders, then we regard it as a bordered cell, and add the bottom and left borders for it.

            // extend borders from upleft to rightdown
            #region four directions
            for (int i = 0; i < Height; i++)
            {
                for (int j = 0; j < Width; j++)
                {
                    if (borderMap[i, j, 1] && borderMap[i, j, 2])
                    {
                        borderMap[i, j, 0] = true;
                        borderMap[i, j, 3] = true;
                        if (j < Width) borderMap[i, j + 1, 2] = true;
                        if (i < Height) borderMap[i + 1, j, 1] = true;
                    }
                }
            }

            // extend borders from upright to downleft
            for (int i = 0; i < Height; i++)
            {
                for (int j = Width - 1; j >= 0; j--)
                {
                    if (borderMap[i, j, 1] && borderMap[i, j, 3])
                    {
                        borderMap[i, j, 0] = true;
                        borderMap[i, j, 2] = true;
                        if (j - 1 >= 0) borderMap[i, j - 1, 3] = true;
                        if (i < Height) borderMap[i + 1, j, 1] = true;
                    }
                }
            }

            // extend borders from downleft to upright
            for (int i = Height - 1; i >= 0; i--)
            {
                for (int j = 0; j < Width; j++)
                {
                    if (borderMap[i, j, 0] == true && borderMap[i, j, 2] == true)
                    {
                        borderMap[i, j, 1] = true;
                        borderMap[i, j, 3] = true;
                        if (j < Width) borderMap[i, j + 1, 2] = true;
                        if (i - 1 >= 0) borderMap[i - 1, j, 0] = false;
                    }
                }
            }
            // extend borders from downright to upleft
            for (int i = Height - 1; i >= 0; i--)
            {
                for (int j = Width - 1; j >= 0; j--)
                {
                    if (borderMap[i, j, 0] && borderMap[i, j, 3])
                    {
                        borderMap[i, j, 1] = true;
                        borderMap[i, j, 2] = true;
                        if (j - 1 >= 0) borderMap[i, j - 1, 3] = true;
                        if (i - 1 >= 0) borderMap[i - 1, j, 0] = true;
                    }
                }
            }
            #endregion
            locateBorderCohensions(borderMap);
        }

        private List<Boundary> generateColorCohensions(double colorHeightThresh = 5, double colorWidthThresh = 3, double nullFeatureAreaThresh = 2)
        {
            // search for the colored squared regions in the sheet
            List<Boundary> colorRegions = new List<Boundary>();
            bool[,] colorMapMark = new bool[Height, Width];

            for (int i = 0; i < Height; i++)
            {
                for (int j = 0; j < Width; j++)
                {
                    // index of featuremap
                    colorMapMark[i, j] = featureMap[i, j].HasFillColor;
                }
            }

            for (int i = 0; i < Height; i++)
            {
                for (int j = 0; j < Width; j++)
                {
                    if (!colorMapMark[i, j]) continue;
                    // find the neighboring colored region (right and down dirctions)
                    int stepRow = 0;
                    int stepCol = 0;
                    // search for the retion in the direction of diagonal first
                    while (i + stepRow < Height && j + stepCol < Width)
                    {
                        if (!colorMapMark[i + stepRow, j + stepCol]) break;
                        stepRow++;
                        stepCol++;
                    }
                    if (stepRow > 0) stepRow--;
                    if (stepCol > 0) stepCol--;
                    while (i + stepRow < Height)
                    {
                        if (!colorMapMark[i + stepRow, j + stepCol]) break;
                        stepRow++;
                    }
                    if (stepRow > 0) stepRow--;
                    while (j + stepCol < Width)
                    {
                        if (!colorMapMark[i + stepRow, j + stepCol]) break;
                        stepCol++;
                    }
                    if (stepCol > 0) stepCol--;
                    //uncolored this colored region
                    for (int p = 0; p <= stepRow; p++)
                    {
                        for (int q = 0; q <= stepCol; q++)
                        {
                            colorMapMark[i + p, j + q] = false;
                        }
                    }

                    if (stepRow >= colorHeightThresh || stepCol >= colorWidthThresh)
                    {
                        Boundary forcedColorRegion = new Boundary(i + 1, i + 1 + stepRow, j + 1, j + 1 + stepCol);
                        colorRegions.Add(forcedColorRegion);
                    }
                }

            }
            return colorRegions;
        }

        private List<Boundary> ConvertMergedRanges(List<Boundary> mergedAreas)
        {
            return mergedAreas.Select(mergedArea => new Boundary(
                mergedArea.top + rowOffset + 1,
                mergedArea.bottom + rowOffset + 1,
                mergedArea.left + colOffset + 1,
                mergedArea.right + colOffset + 1)).ToList();
        }

        private List<Boundary> GenerateMergedCohensions(ref List<int> rowLines, ref List<int> colLines)
        {
            List<Boundary> mergeBoxesAll = new List<Boundary>();
            mergeBoxesAll.AddRange(mergeBoxes);

            // unify the overlapping and neighboring boxes
            ////List<Boundary> unifiedRanges = Utils.GetUnifiedRanges(mergeBoxes);
            ////mergeBoxesAll.AddRange(unifiedRanges);

            return mergeBoxesAll;
        }
        #endregion


        #region compute value maps, border, content, color, diff
        private void computeValueDiff()
        {
            //the differece for the two adjacent cells in the row and column directions
            rowDiffs = new List<List<double>>();
            colDiffs = new List<List<double>>();

            // whether is different for the two adjacent cells in the row and column directions
            rowDiffsBool = new List<List<bool>>();
            colDiffsBool = new List<List<bool>>();

            //initialize rowDiffBool and rowDiffs
            for (int i = 0; i < Height - 1; i++)
            {
                List<bool> rowDiffBool = new List<bool>();
                List<double> rowDiff = new List<double>();
                for (int k = 0; k < Width; k++)
                {
                    rowDiffBool.Add(false);
                    rowDiff.Add(0);
                }
                rowDiffsBool.Add(rowDiffBool);
                rowDiffs.Add(rowDiff);
            }

            //initialize colDiffBool and colDiffs
            for (int i = 0; i < Width - 1; i++)
            {
                List<bool> colDiffBool = new List<bool>();
                List<double> colDiff = new List<double>();
                for (int k = 0; k < Height; k++)
                {
                    colDiffBool.Add(false);
                    colDiff.Add(0);
                }
                colDiffsBool.Add(colDiffBool);
                colDiffs.Add(colDiff);
            }

            //  calculate rowDiffBool and rowDiffs, colDiffBool and colDiffs
            for (int i = 0; i < Height - 1; i++)
            {
                for (int j = 0; j < Width - 1; j++)
                {

                    // calculate rowDiffBool and rowDiffs
                    if (i > 0)
                    {
                        // up down windows
                        Boundary boxDown = new Boundary(i + 1, i + 1, 1, Width - 1);
                        Boundary boxUp = new Boundary(i, i, 1, Width - 1);
                        // sum up values in the up and down windows

                        int sumDownValue = sumAll.SubmatrixSum(boxDown);
                        int sumUpValue = sumAll.SubmatrixSum(boxUp);

                        int sumDownContentExsit = sumContentExist.SubmatrixSum(boxDown);
                        int sumUpContentExsit = sumContentExist.SubmatrixSum(boxUp);

                        if (sumUpValue == sumDownValue)
                        {
                            rowDiffsBool[i - 1][j] = false;
                        }
                        //  Whether or not exist content
                        else if (Utils.VerifyValuesOpposite(valueMapContent[i, j], valueMapContent[i - 1, j]) && Utils.VerifyValuesOpposite(sumDownContentExsit, sumUpContentExsit))
                        {
                            rowDiffsBool[i - 1][j] = true;
                        }
                        // content ununiform 
                        else if (Utils.VerifyValuesDiff(sumDownContentExsit, sumUpContentExsit, 5))
                        {
                            rowDiffsBool[i - 1][j] = true;
                        }
                        //color difference
                        else if (Utils.VerifyValuesOpposite(valueMapColor[i, j], valueMapColor[i - 1, j]))
                        {
                            rowDiffsBool[i - 1][j] = true;
                        }
                        //border difference
                        else if (featureMap[i, j].HasTopBorder && (!featureMap[i + 1, j].HasTopBorder || !featureMap[i - 1, j].HasTopBorder))
                        {
                            rowDiffsBool[i - 1][j] = true;
                        }
                        else
                        {
                            rowDiffsBool[i - 1][j] = false;
                        }
                        // calculate rowDiffs directly use sumDownValue and sumUpValue
                        rowDiffs[i - 1][j] = sumDownValue / 5 - sumUpValue / 5;

                    }
                    // calculate colDiffBool and colDiffs
                    if (j > 0)
                    {
                        Boundary boxRight = new Boundary(1, Height - 1, j + 1, j + 1);
                        Boundary boxLeft = new Boundary(1, Height - 1, j, j);
                        int sumRightValue = sumAll.SubmatrixSum(boxRight);
                        int sumLeftValue = sumAll.SubmatrixSum(boxLeft);
                        int sumRightContentExsit = sumContentExist.SubmatrixSum(boxRight);
                        int sumLeftContentExsit = sumContentExist.SubmatrixSum(boxLeft);
                        if (sumRightValue == sumLeftValue)
                        {
                            colDiffsBool[j - 1][i] = false;
                        }
                        else if (Utils.VerifyValuesOpposite(valueMapContent[i, j], valueMapContent[i, j - 1]) && Utils.VerifyValuesOpposite(sumLeftContentExsit, sumRightContentExsit))
                        {
                            colDiffsBool[j - 1][i] = true;
                        }
                        else if (Utils.VerifyValuesDiff(sumLeftContentExsit, sumRightContentExsit, 5))
                        {
                            colDiffsBool[j - 1][i] = true;
                        }
                        else if (Utils.VerifyValuesOpposite(valueMapColor[i, j], valueMapColor[i, j - 1]))
                        {
                            colDiffsBool[j - 1][i] = true;
                        }
                        else if (featureMap[i, j].HasLeftBorder && (!featureMap[i, j + 1].HasLeftBorder || !featureMap[i, j - 1].HasLeftBorder))
                        {
                            rowDiffsBool[i - 1][j] = true;
                        }
                        else
                        {
                            colDiffsBool[j - 1][i] = false;
                        }
                        colDiffs[j - 1][i] = sumRightValue / 5 - sumLeftValue / 5;
                    }

                }

            }
        }

        private int[,] ComputeValueMap(Func<CellFeatures, int> calcFunc)
            => ComputeValueMap((i, j) => calcFunc(featureMap[i, j]));

        private int[,] ComputeValueMap(Func<int, int, int> calcFunc)
        {
            valueMapAll = new int[Height, Width];
            for (int i = 0; i < Height; i++)
                for (int j = 0; j < Width; j++)
                    valueMapAll[i, j] = calcFunc(i, j);

            return valueMapAll;
        }
        #endregion

        #region value range sums

        public static int ValueSumRange(List<Boundary> boxes, int[,] valueMapSum)
        {
            // the overall area of boxes with overlaps
            int result = 0;
            List<Boundary> boxesCalculated = new List<Boundary>();
            foreach (var box in boxes)
            {
                // if one box is contained by another, then just skip this one
                if (Utils.ContainsBox(boxes, box)) continue;

                result += valueMapSum.SubmatrixSum(box);
                // subtract the overlaps by more than two boxes
                foreach (var boxIn in boxesCalculated)
                {
                    if (Utils.isOverlap(boxIn, box) && !boxIn.Equals(box))
                    {
                        Boundary boxOverlaped = Utils.OverlapBox(boxIn, box);
                        result -= valueMapSum.SubmatrixSum(boxOverlaped);
                    }
                }
                // subtract the overlaps by more than three boxes
                foreach (var boxIn1 in boxesCalculated)
                {
                    foreach (var boxIn2 in boxesCalculated)
                    {
                        if (Utils.isOverlap(boxIn1, box) && Utils.isOverlap(boxIn2, box) && Utils.isOverlap(boxIn2, boxIn1) && !boxIn2.Equals(boxIn1) && !boxIn1.Equals(box) && !boxIn2.Equals(box))
                        {
                            Boundary boxOverlaped = Utils.OverlapBox(boxIn1, boxIn2, box);
                            result += valueMapSum.SubmatrixSum(boxOverlaped);
                        }
                    }
                }
                // Omitted the more complicated cases for simiplicity, for example, subtract the overlaps by more than four boxes

                boxesCalculated.Add(box);
            }
            return result;
        }

        public double ContentExistValueDensity(Boundary box)
        {
            if (box.bottom < box.top || box.right < box.left) return 0;
            return sumContentExist.SubmatrixSum(box) / Utils.AreaSize(box);
        }

        public double RowContentExistValueDensitySplit(Boundary box, double split = 4)
        {
            if (box.bottom < box.top || box.right < box.left) return 0;
            double stride = (box.right - box.left + 1) / split;
            double cntExists = 0;
            for (int i = 0; i < split; i++)
            {
                if (sumContentExist.SubmatrixSum(new Boundary(box.top, box.bottom, Convert.ToInt32(box.left + i * stride), Convert.ToInt32(box.left + stride * (i + 1)) - 1)) > 0)
                {
                    cntExists++;
                }
            }
            return cntExists / split;
        }

        public double ColContentExistValueDensitySplit(Boundary box, double split = 4)
        {
            if (box.bottom < box.top || box.right < box.left) return 0;
            double stride = (box.bottom - box.top + 1) / split;
            double cntExists = 0;
            for (int i = 0; i < split; i++)
            {
                if (sumContentExist.SubmatrixSum(new Boundary(Convert.ToInt32(box.top + i * stride), Convert.ToInt32(box.top + stride * (i + 1)) - 1, box.left, box.right)) > 0)
                {
                    cntExists++;
                }
            }
            return cntExists / split;
        }
        #endregion

        #region row/col relation computation
        public bool verifyMutualMatchCol(Boundary box1, Boundary box2)
        {
            int up = box1.top;
            int down = box1.bottom;
            for (int i = up; i <= down; i++)
            {
                if (Math.Min(2, valueMapContent[i - 1, box1.left - 1]) != Math.Min(2, valueMapContent[i - 1, box2.left - 1]))
                {
                    return false;
                }
            }
            return true;
        }

        public double ComputeSimilarRow(Boundary box1, Boundary box2)
        {
            List<double> similars = new List<double>();

            for (int i = box1.top; i <= box1.bottom; i++)
            {
                for (int j = box2.top; j <= box2.bottom; j++)
                {
                    int left = box1.left;
                    int right = box1.right;
                    Boundary window1 = new Boundary(i, i, left, left);
                    Boundary window2 = new Boundary(j, j, left, left);
                    while (sumContentExist.SubmatrixSum(window1) == 0 && sumContentExist.SubmatrixSum(window2) == 0 && left <= right)
                    {
                        left++;
                        window1 = new Boundary(i, i, left, left);
                        window2 = new Boundary(j, j, left, left);
                    }
                    window1 = new Boundary(i, i, right, right);
                    window2 = new Boundary(j, j, right, right);
                    while (sumContentExist.SubmatrixSum(window1) == 0 && sumContentExist.SubmatrixSum(window2) == 0 && left <= right)
                    {
                        right--;
                        window1 = new Boundary(i, i, right, right);
                        window2 = new Boundary(j, j, right, right);
                    }
                    window1 = new Boundary(i, i, left, right);
                    window2 = new Boundary(j, j, left, right);

                    if (window1.Equals(window2))
                    {
                        continue;
                    }
                    List<double> list1_1 = convertFeatureToList(window1, feature => feature.AlphabetRatio);
                    List<double> list2_1 = convertFeatureToList(window2, feature => feature.AlphabetRatio);
                    List<double> list1_2 = convertFeatureToList(window1, feature => feature.NumberRatio);
                    List<double> list2_2 = convertFeatureToList(window2, feature => feature.NumberRatio);

                    if (sumContentExist.SubmatrixSum(window1) <= 2 || sumContentExist.SubmatrixSum(window1) / list1_1.Count <= 0.3
                        || sumContentExist.SubmatrixSum(window2) <= 2 || sumContentExist.SubmatrixSum(window2) / list1_2.Count <= 0.3)
                    {
                        continue;
                    }

                    similars.Add(Utils.r1(list1_1, list2_1));
                    similars.Add(Utils.r1(list1_2, list2_2));
                }
            }
            if (similars.Count == 0)
                return 1;
            return Utils.average(similars);
        }

        private List<double> convertFeatureToList(Boundary box, Func<CellFeatures, double> featureFunc)
        {
            List<double> ListedArr = new List<double>();
            for (int i = box.top - 1; i <= box.bottom - 1; i++)
            {
                for (int j = box.left - 1; j <= box.right - 1; j++)
                {
                    ListedArr.Add(featureFunc(featureMap[i, j]));
                }
            }
            return ListedArr;
        }

        #endregion

        #region formula analysis
        private List<Boundary>[,] preProcessFormulas(string[,] formus)
        {
            int height = formus.GetLength(0);
            int width = formus.GetLength(1);

            var formulaReferenceRanges = new List<Boundary>[height, width];

            // Regular expressions for date and lookup
            Regex RegLookup = new Regex("LOOKUP", RegexOptions.Compiled);
            Regex RegDate = new Regex("DATE", RegexOptions.Compiled);

            // Traverse all the formulas in the sheet
            for (int row = 0; row < height; row++)
            {
                // rowReferRanges,  one reference ranges for each cell in a row
                List<List<Boundary>> rowReferRanges = new List<List<Boundary>>();

                for (int col = 0; col < width; col++)
                {
                    List<Boundary> ranges = new List<Boundary>();
                    // the first kind of formula, like "SUM(A10:B13)", but not "R1C10"
                    if (formus[row, col] != null && formus[row, col] != "")
                    {
                        string formula = formus[row, col];
                        // if formula is long, just cut it for simplility
                        if (formula.Length > 50) formula = formula.Substring(0, 50);
                        // when  formula is long, then it seems hard to analysis correctly, "=SUMIF(mSRPV!$A:$A,$C12,mSRPV!J:J)"
                        // when  formula contains dates, then it may be important and should not be missed, "=DATE(YEAR(L10),MONTH(L10)+1,1)"
                        // when formula contains lookup, then need be careful, because the result region often refer to the wrong places.
                        MatchCollection mcsDate = RegDate.Matches(formula);
                        MatchCollection mcsLookup = RegLookup.Matches(formula);
                        if (mcsDate.Count != 0 || (mcsLookup.Count == 0 && formula.Length <= 18))
                        {
                            try
                            {
                                ranges = formulaAnalysis(formula);
                            }
                            catch
                            {
                            }
                        }
                        else
                        {
                            // refer to a empty cell as default 
                            ranges = new List<Boundary>();
                            ranges.Add(new Boundary(1, 1, 1, 1));
                        }

                    }

                    formulaReferenceRanges[row, col] = ranges;
                }
            }

            return formulaReferenceRanges;
        }

        private Boundary formulaRegionAnalysis(string regionStr)
        {   // Regular expressions
            Regex RegRow = new Regex("\\d+");
            Regex RegCol = new Regex("[A-Z]+");
            Boundary box = new Boundary(0, 0, 0, 0);

            MatchCollection mcRow = RegRow.Matches(regionStr);
            MatchCollection mcCol = RegCol.Matches(regionStr);

            if (mcRow.Count == 2 && mcCol.Count == 2)
            {
                string rowStr = mcRow[0].ToString();
                string colStr = mcCol[0].ToString();
                box.top = Convert.ToInt32(rowStr);
                box.left = colStr.Select(t => t - 'A' + 1).Aggregate(0, (current, temp) => temp + current * 26);

                rowStr = mcRow[1].ToString();
                colStr = mcCol[1].ToString();
                box.bottom = Convert.ToInt32(rowStr);
                box.right = colStr.Select(t => t - 'A' + 1).Aggregate(0, (current, temp) => temp + current * 26);

            }

            else if (mcCol.Count == 2 && mcRow.Count == 0)
            {
                box = new Boundary(1, Height, 0, 0);
                string colStr = mcCol[0].ToString();
                box.left = colStr.Select(t => t - 'A' + 1).Aggregate(0, (current, temp) => temp + current * 26);

                colStr = mcCol[1].ToString();
                box.right = colStr.Select(t => t - 'A' + 1).Aggregate(0, (current, temp) => temp + current * 26);

            }
            else if (mcRow.Count == 2 && mcCol.Count == 0)
            {
                box = new Boundary(0, 0, 1, Width);

                string rowStr = mcRow[0].ToString();
                box.top = Convert.ToInt32(rowStr);
                rowStr = mcRow[1].ToString();
                box.bottom = Convert.ToInt32(rowStr);
            }
            else if (mcRow.Count == 1 && mcCol.Count == 1)
            {
                box = new Boundary(0, 0, 0, 0);
                string rowStr = mcRow[0].ToString();
                string colStr = mcCol[0].ToString();
                box.top = Convert.ToInt32(rowStr);
                box.left = colStr.Select(t => t - 'A' + 1).Aggregate(0, (current, temp) => temp + current * 26);
                box.bottom = box.top;
                box.right = box.left;
            }
            else
            {
                return box;
            }

            box.top = Math.Max(box.top + rowOffset, 1 + rowOffset);
            box.bottom = Math.Min(box.bottom + rowOffset, Height - rowOffset);
            box.left = Math.Max(box.left + colOffset, 1 + colOffset);
            box.right = Math.Min(box.right + colOffset, Width - colOffset);

            return box;
        }
        private List<Boundary> formulaAnalysis(string formula)
        {
            // Regular expressions
            Regex RegRegion = new Regex("\\$?[A-Z]+\\$?\\d+:\\$?[A-Z]+\\$?\\d+");
            Regex RegCols = new Regex("\\$?[A-Z]+:\\$?[A-Z]+");
            Regex RegRows = new Regex("\\$?\\d+:\\$?\\d+");
            Regex RegCell = new Regex("\\$?[A-Z]+\\$?\\d+");

            List<Boundary> ranges = new List<Boundary>();

            Dictionary<string, bool> records = new Dictionary<string, bool>();

            // first analyze region,  then col/row, cell last

            //region level reference
            foreach (var mc in RegRegion.Matches(formula))
            {
                string regionStr = mc.ToString();
                if (records.ContainsKey(regionStr)) continue;
                else records[regionStr] = true;

                Boundary box = formulaRegionAnalysis(regionStr);

                if (box.top <= box.bottom && box.left <= box.right && box.top * box.bottom * box.left * box.right != 0)
                {
                    ranges.Add(box);
                }
            }
            if (ranges.Count > 0)
            {
                return ranges;
            }

            // col  level reference
            foreach (var mc in RegCols.Matches(formula))
            {
                string regionStr = mc.ToString();
                if (records.ContainsKey(regionStr)) continue;
                else records[regionStr] = true;

                Boundary box = formulaRegionAnalysis(regionStr);

                if (box.top <= box.bottom && box.left <= box.right && box.top * box.bottom * box.left * box.right != 0)
                {
                    ranges.Add(box);
                }
            }

            // row  level reference
            foreach (var mc in RegRows.Matches(formula))
            {
                string regionStr = mc.ToString();
                if (records.ContainsKey(regionStr)) continue;
                else records[regionStr] = true;

                Boundary box = formulaRegionAnalysis(regionStr);

                if (box.top <= box.bottom && box.left <= box.right && box.top * box.bottom * box.left * box.right != 0)
                {
                    ranges.Add(box);
                }
            }
            if (ranges.Count > 0)
            {
                return ranges;
            }

            // cell level reference
            foreach (var mc in RegCell.Matches(formula))
            {
                string regionStr = mc.ToString();
                if (records.ContainsKey(regionStr)) continue;
                else records[regionStr] = true;

                Boundary box = formulaRegionAnalysis(regionStr);

                if (box.top <= box.bottom && box.left <= box.right && box.top * box.bottom * box.left * box.right != 0)
                {
                    ranges.Add(box);
                }
            }

            return ranges;
        }

        private void DealFormulaRanges(List<Boundary> referRanges, int cellHeight, int cellWidth, int recurseDepth, int recurseMaxDepth = 10)
        {
            // update feature and content regarding to formulas, with recursive method
            int i = cellHeight - 1;
            int j = cellWidth - 1;
            if (contentStrs[i, j] != "" && contentStrs[i, j] != CellFeatures.DefaultContentForFomula)
            {
                return;
            }
            contentStrs[i, j] = contentStrs[i, j] + "_index_" + i.ToString() + "_" + j.ToString();
            bool markFindout = false;
            // tranverse all the reference ranges
            foreach (var range in referRanges)
            {
                // if already find out the target feature and content, then break
                if (markFindout) break;
                // tranverse all the cells in this range
                for (int row = range.top; row <= range.bottom; row++)
                {
                    if (markFindout) break;
                    for (int col = range.left; col <= range.right; col++)
                    {
                        #region update the feature and content for cells in the recursive path
                        if (row < 1 || row > Height || col < 1 || col > Width) continue;
                        // if text for the reference cell is not none
                        if (!featureMap[row - 1, col - 1].MarkText) continue;
                        if (recurseDepth < recurseMaxDepth && formulaRanges[row - 1, col - 1].Count > 0
                            && (contentStrs[row - 1, col - 1] == CellFeatures.DefaultContentForFomula || @Regex.Split(contentStrs[row - 1, col - 1], "_index_")[0] == CellFeatures.DefaultContentForFomula))
                        {
                            DealFormulaRanges(formulaRanges[row - 1, col - 1], row, col, recurseDepth + 1);
                        }

                        featureMap[i, j].TextLength = featureMap[row - 1, col - 1].TextLength;
                        featureMap[i, j].AlphabetRatio = featureMap[row - 1, col - 1].AlphabetRatio;
                        featureMap[i, j].NumberRatio = featureMap[row - 1, col - 1].NumberRatio;
                        featureMap[i, j].SpCharRatio = featureMap[row - 1, col - 1].SpCharRatio;

                        //add suffix for content text, it can avoid dead loop, and it can make sure variaty for cell values.
                        contentStrs[i, j] = contentStrs[i, j] + "_retrieved";

                        //if find a valid text, then end the tranverse
                        if (@Regex.Split(contentStrs[row - 1, col - 1], "_index_")[0] != ""
                            && @Regex.Split(contentStrs[row - 1, col - 1], "_index_")[0] != CellFeatures.DefaultContentForFomula)
                        {
                            markFindout = true;
                            break;
                        }
                        #endregion
                    }
                }
            }
        }

        private void UpdateFeatAndContentsByFormula()
        {
            // update the feature map for the sheet by the formulas 

            Random rand = new Random(DateTime.Now.Millisecond);

            for (int row = rowOffset; row < Height - rowOffset; row++)
            {
                for (int col = colOffset; col < Width - colOffset; col++)
                {
                    List<Boundary> ranges = formulaRanges[row, col];
                    if (ranges.Count > 0)
                    {
                        DealFormulaRanges(ranges, row + 1, col + 1, 0);
                    }
                }
            }

            for (int row = rowOffset; row < Height - rowOffset; row++)
            {
                for (int col = colOffset; col < Width - colOffset; col++)
                {
                    if (contentStrs[row, col] == CellFeatures.DefaultContentForFomula)
                    {
                        contentStrs[row, col] = contentStrs[row, col] + "_index_" + row.ToString() + "_" + col.ToString();
                    }
                }
            }
        }
        #endregion

        #region findout headers 
        public List<Boundary> FindoutUpheaders(TableDetectionHybrid detector, List<Boundary> regions)
        {
            List<Boundary> upHeaders = new List<Boundary>();
            for (int i = 0; i < regions.Count; i++)
            {
                Boundary box1 = regions[i];
                for (int k = 0; k < 5; k++)
                {
                    Boundary upHeaderCandidate = Utils.UpRow(box1, start: k);
                    if ((sumContentExist.SubmatrixSum(upHeaderCandidate) >= 6
                        && TextDistinctCount(upHeaderCandidate) > 1
                        && ContentExistValueDensity(upHeaderCandidate) >= 2 * 0.5)
                        || (sumContentExist.SubmatrixSum(upHeaderCandidate) >= 4
                        && TextDistinctCount(upHeaderCandidate) > 1
                        && ContentExistValueDensity(upHeaderCandidate) >= 2 * 1))
                    {
                        if (detector.IsHeaderUp(upHeaderCandidate) && ContentExistValueDensity(upHeaderCandidate) > 2 * 0.7)
                        {
                            upHeaders.Add(upHeaderCandidate);
                        }
                        break;
                    }
                }

            }
            return Utils.DistinctBoxes(upHeaders);
        }

        public List<Boundary> FindoutLeftheaders(TableDetectionHybrid detector, List<Boundary> regions)
        {
            List<Boundary> leftHeaders = new List<Boundary>();
            for (int i = 0; i < regions.Count; i++)
            {
                Boundary box1 = regions[i];
                for (int k = 0; k < 5; k++)
                {
                    Boundary leftHeaderCandidate = Utils.LeftCol(box1, start: k);
                    if (sumContentExist.SubmatrixSum(leftHeaderCandidate) >= 6 && TextDistinctCount(leftHeaderCandidate) > 1
                        && ContentExistValueDensity(leftHeaderCandidate) >= 2 * 0.5)
                    {
                        if (detector.IsHeaderLeft(leftHeaderCandidate))
                        {
                            leftHeaders.Add(leftHeaderCandidate);
                        }
                        break;
                    }
                }

            }
            return Utils.DistinctBoxes(leftHeaders);
        }
        #endregion
        public bool ExistsMerged(Boundary box)
        {
            return mergeBoxes.Any(range => Utils.isOverlap(range, box));
        }

        public int TextDistinctCount(Boundary box)
        {
            var hashSet = new HashSet<string>();
            for (int i = box.top; i <= box.bottom; i++)
            {
                for (int j = box.left; j <= box.right; j++)
                {
                    if (featureMap[i - 1, j - 1].MarkText)
                    {
                        hashSet.Add(contentStrs[i - 1, j - 1]);
                    }
                }
            }

            return hashSet.Count;
        }
        private int CntNoneBorder(Boundary box)
        {
            Boundary boxUp = new Boundary(box.top, box.top, box.left, box.right);
            Boundary boxDown = new Boundary(box.bottom, box.bottom, box.left, box.right);
            Boundary boxLeft = new Boundary(box.top, box.bottom, box.left, box.left);
            Boundary boxRight = new Boundary(box.top, box.bottom, box.right, box.right);
            return new[] { boxUp, boxDown, boxLeft, boxRight }
                .Count(box1 => sumContent.SubmatrixSum(box1) == 0 && sumColor.SubmatrixSum(box1) == 0);
        }
    }
}
