#!/usr/bin/env bash

# bash mlsum.sh train en en mspm4_mlsum debug --encoder-version v1 --adapter-num-layer 2 --batch-size 8 --ln-after-proj;

# bash mlsum.sh generate en zh mspm4_xgiga debug _best

MODE="$1"
LG="$2"
EVAL_LG="$3" # 'all' or single language
DATAVER="$4"
NAME="$5"

declare -i args_start=6
if [ $MODE == "generate" ]; then
    args_start=$((${args_start}+1))
fi

argslist=""
for (( i = ${args_start}; i <= $# ; i++ ))
    do
        j=${!i}
        argslist="${argslist} $j "
    done
echo $argslist >&2

predicted_lgs=(${EVAL_LG})
if [ ${EVAL_LG} == 'all' ]; then
    predicted_lgs=(en zh fr)
fi

predicted_lg_str=""
for lg in ${predicted_lgs[*]}
do
    predicted_lg_str="$predicted_lg_str $lg"
done
echo "predicted_lg_str: " ${predicted_lg_str}

renameSplit(){
    path=$1
    oldsplit=$2
    newsplit=$3

    mv ${path}/${oldsplit}.doc-sum.doc.bin ${path}/${newsplit}.doc-sum.doc.bin
    mv ${path}/${oldsplit}.doc-sum.doc.idx ${path}/${newsplit}.doc-sum.doc.idx
    mv ${path}/${oldsplit}.doc-sum.sum.bin ${path}/${newsplit}.doc-sum.sum.bin
    mv ${path}/${oldsplit}.doc-sum.sum.idx ${path}/${newsplit}.doc-sum.sum.idx

    echo "rename ${path}/${oldsplit}.* to ${path}/${newsplit}.*"
}

cd "$(dirname $0)" || return
echo "Install fairseq" >&2

export https_proxy=http://bj-rd-proxy.byted.org:3128 http_proxy=http://bj-rd-proxy.byted.org:3128 no_proxy=code.byted.org

sudo mkdir -p /usr/lib/python3.7/site-packages/
sudo pip3 install -e fairseq
bash install_pyrouge.sh

export PYROUGE_HOME_DIR=$(pwd)/RELEASE-1.5.5
export PYROUGE_TEMP_PATH=/opt/tiger

pyrouge_set_rouge_path $PYROUGE_HOME_DIR
chmod +x $PYROUGE_HOME_DIR/ROUGE-1.5.5.pl

prefix=hdfs://haruna/home/byte_arnold_hl_mlnlc/user/wuxianze.0
dataset_path=${prefix}/Datasets/multilingual/data-bin/${DATAVER}/${LG}
tensorboard_logdir=${prefix}/Workspace/Multilingual/mlsum_${LG}/logs/${NAME}
checkpoint_path=${prefix}/Workspace/Multilingual/mlsum_${LG}/checkpoints/${NAME}
pretrained_path=${prefix}/Workspace/Multilingual/pretrained

# change checkpoint path
hdfs dfs -mkdir -p $tensorboard_logdir
hdfs dfs -mkdir -p $checkpoint_path

local_root=~/mlsum_${NAME}
resource_root=${local_root}/resource
output_path=${local_root}/output
model_path=${local_root}/model
adapter_path=${local_root}/adapter
component_path=~/component
mkdir -p ${resource_root}
mkdir -p ${output_path}
mkdir -p ${model_path}
mkdir -p ${adapter_path}
mkdir -p ${component_path}

local_dataset_path=${resource_root}/dataset
mkdir -p ${local_dataset_path}
hadoop fs -copyToLocal ${dataset_path}/* ${local_dataset_path}
echo "Download resource from ${dataset_path} to ${local_dataset_path}" >&2

renameSplit ${local_dataset_path} "valid" "valid_$LG"
renameSplit ${local_dataset_path} "test" "test_$LG"

for lg in ${predicted_lgs[*]}
do
    if [ ! -e ${local_dataset_path}}/valid_${lg}.doc-sum.doc.bin ]; then
        echo "Download resource from ${prefix}/Datasets/multilingual/data-bin/${DATAVER}/$lg/valid.* to ${local_dataset_path}/" >&2
        hdfs dfs -get ${prefix}/Datasets/multilingual/data-bin/${DATAVER}/$lg/valid.* ${local_dataset_path}/
        renameSplit ${local_dataset_path} "valid" "valid_$lg"
    fi
done

local_tensorboard_path=${output_path}/tensorboard_logdir
mkdir -p ${local_tensorboard_path}

local_checkpoint_path=${output_path}/checkpoint_path
mkdir -p ${local_checkpoint_path}

local_pretrained_path=${model_path}/mbart.cc25.v2
if [ ! -d ${local_pretrained_path} ]; then
    echo "Load pretrained model from ${pretrained_path}/mbart.cc25.v2.tar.gz to ${local_pretrained_path}" >&2
    hadoop fs -copyToLocal ${pretrained_path}/mbart.cc25.v2.tar.gz ${model_path}
    tar -xvzf ${model_path}/mbart.cc25.v2.tar.gz -C ${model_path}
else
    echo "Pretrained model in ${local_pretrained_path}" >&2
fi

echo "Finish download files" >&2

langs=ar_AR,cs_CZ,de_DE,en_XX,es_XX,et_EE,fi_FI,fr_XX,gu_IN,hi_IN,it_IT,ja_XX,kk_KZ,ko_KR,lt_LT,lv_LV,my_MM,ne_NP,nl_XX,ro_RO,ru_RU,si_LK,tr_TR,vi_VN,zh_CN

if [[ "$MODE" == "train" || "$MODE" == "train_2stage" ]]; then
    echo "Training..."

    (inotifywait -m ${local_checkpoint_path} -e close_write |
        while read path action file; do
                if [[ "$file" =~ .*pt$ ]]; then
                    echo "Checkpoint detected: $file" >&2
                    hadoop fs -put -f ${local_checkpoint_path}/$file ${checkpoint_path}/ && echo "checkpoint uploaded: $file to ${checkpoint_path}/$file" >&2
                    rm ${local_checkpoint_path}/$file
                fi
        done) &

    restore_file=${local_pretrained_path}/model.pt

    valid_subsets=
    for lg in ${predicted_lgs[*]}
    do
        if [ "${valid_subsets}" == "" ]; then
            valid_subsets="valid_${lg}"
        else
            valid_subsets="${valid_subsets},valid_${lg}"
        fi
    done

    python3 fairseq/train.py ${local_dataset_path} --ddp-backend=no_c10d \
        --save-dir ${local_checkpoint_path} \
        --tensorboard-logdir ${local_tensorboard_path} \
        --restore-file ${restore_file} \
        --task summarization_from_pretrained_mbart_mspm4 \
        --arch mbart_summ_abs_large \
        --source-lang doc --target-lang sum \
        --langs $langs \
        --dataset-impl mmap \
        --truncate-source \
        --encoder-normalize-before --decoder-normalize-before \
        --layernorm-embedding \
        --criterion label_smoothed_cross_entropy --label-smoothing 0.2 \
        --required-batch-size-multiple 1 \
        --dropout 0.1 --attention-dropout 0.1 \
        --weight-decay 0.01 --optimizer adam \
        --lr 3e-5 --min-lr -1 \
        --lr-scheduler polynomial_decay \
        --clip-norm 0.1 \
        --skip-invalid-size-inputs-valid-test \
        --find-unused-parameters \
        --num-workers 10  --adam-betas '(0.9, 0.999)' --adam-eps 1e-08 \
        --total-num-update 358900 --warmup-updates 17945 \
        --max-epoch 20 --fp16 \
        --log-format simple \
        --keep-best-checkpoints 3 \
        --user-dir examples/summarization \
        --valid-subset ${valid_subsets} \
        --log-interval 1600 \
        --loose-load \
        --max-sentences-valid 8 \
        --patience 4 \
        --reset-optimizer --reset-dataloader --reset-meters --reset-lr-scheduler \
        $argslist
    
    echo "Put ${local_tensorboard_path} to ${tensorboard_logdir}" >&2
    hadoop fs -put -f ${local_tensorboard_path}/* ${tensorboard_logdir}/
    sleep 300

elif [ "$MODE" == "generate" ]; then
    ckpt_suffix="$6"

    if [ ! -e ${local_dataset_path}}/test_${EVAL_LG}.doc-sum.doc.bin ]; then
        echo "Download resource from ${prefix}/Datasets/multilingual/data-bin/${DATAVER}/${EVAL_LG}/test.* to ${local_dataset_path}/" >&2
        hdfs dfs -get ${prefix}/Datasets/multilingual/data-bin/${DATAVER}/${EVAL_LG}/test.* ${local_dataset_path}/
        echo "Download resource from ${prefix}/Datasets/multilingual/data-bin/${DATAVER}/${EVAL_LG}/dict.* to ${local_dataset_path}/" >&2
        hdfs dfs -get ${prefix}/Datasets/multilingual/data-bin/${DATAVER}/${EVAL_LG}/dict.* ${local_dataset_path}/
        renameSplit ${local_dataset_path} "test" "test_${EVAL_LG}"
    fi
    
    hadoop fs -copyToLocal ${tensorboard_logdir}/* ${local_tensorboard_path}/

    echo "Generating..."

    rm -r ${local_checkpoint_path}/checkpoint${ckpt_suffix}.pt
    hadoop fs -copyToLocal ${checkpoint_path}/checkpoint${ckpt_suffix}.pt ${local_checkpoint_path}
    echo "Load checkpoints from ${checkpoint_path}/checkpoint${ckpt_suffix}.pt to ${local_checkpoint_path}" >&2

    suffix=$(echo "$argslist" | sed -e "s/-//g"  -e "s/  */_/g")
    python3 fairseq/generate.py ${local_dataset_path} \
        --path ${local_checkpoint_path}/checkpoint${ckpt_suffix}.pt \
        --task summarization_from_pretrained_mbart_mspm4 \
        --gen-subset test_${EVAL_LG} \
        --source-lang doc --target-lang sum \
        --langs $langs \
        --remove-bpe 'sentencepiece' \
        --max-len-a 0.2 --max-len-b 2 --min-len 2 \
        --lenpen 1 \
        --beam 1 \
        --no-repeat-ngram-size 3 \
        --truncate-source \
        --user-dir examples/summarization \
        --batch-size 8 \
        $argslist > ${local_tensorboard_path}/"output$suffix"

        cat ${local_tensorboard_path}/"output$suffix" | grep -P "^H" | \
        sort -V |cut -f 3- | sed -e "s/\[[a-z]\{2\}_[A-Z]\{2\}\]//g" | sed -e "s/\[[a-z]\{,10\}\] //g" > ${local_tensorboard_path}/"test$suffix.hypo"

    echo "Load ground truth file from ${prefix}/Datasets/multilingual/mlsum/${EVAL_LG}/test.${EVAL_LG}.sum"
    hadoop fs -get ${prefix}/Datasets/multilingual/mlsum/${EVAL_LG}/test.${EVAL_LG}.sum ${local_dataset_path}

    hypo=${local_tensorboard_path}/"test$suffix.hypo"
    ref=${local_dataset_path}/test.${EVAL_LG}.sum

    if [ ${EVAL_LG} == "zh" ]; then
    # split the reference and hypothesis into chars
        cat ${hypo} | python3 -u ./xnlg/zh_split_words.py > ${local_tensorboard_path}/"test$suffix.hypo.char"
        cat ${ref} | python3 -u ./xnlg/zh_split_words.py > ${local_dataset_path}/test.y.${EVAL_LG}.char
        hypo=${local_tensorboard_path}/"test$suffix.hypo.char"
        ref=${local_dataset_path}/test.y.${EVAL_LG}.char
        python3 ./xnlg/calc_rouge.py --ref ${ref} --hyp ${hypo} --zh True
    else
        python3 ./xnlg/calc_rouge.py --ref ${ref} --hyp ${hypo}
    fi
    echo "Put ${local_tensorboard_path} to ${tensorboard_logdir}" >&2
    hadoop fs -put -f ${local_tensorboard_path}/* ${tensorboard_logdir}/
    sleep 120
fi
