"""Read and Parse each table-descriptive-sentence as a block."""


from processing.annotation import _TABLE_DESC_SENT_ID, _TABLE_DESC_SENT, _COLOR_WHITE


def strip_block(block, worksheet):
    """Strip off the empty lines before and after the block.
    Note: (s, e) bot inclusive.
    """
    s, e = block    

    new_s = s
    while (new_s < e) and (not worksheet.cell(new_s, 1).value): 
        new_s += 1
    
    new_e = e
    while (new_e > new_s) and (not worksheet.cell(new_e, 1).value):
        new_e -= 1
    
    if new_s >= new_e:
        return None
    else:
        return (new_s, new_e)


def start_block(block, worksheet):
    """Assert the left header of the starting two-line of block."""

    s, e = block
    assert worksheet.cell(s, 1).value.strip() == _TABLE_DESC_SENT_ID
    assert worksheet.cell(s + 1, 1).value.strip() == _TABLE_DESC_SENT

    table_desc_sent_id = worksheet.cell(s, 2).value     # type: int
    table_desc_sent = worksheet.cell(s + 1, 2).value    # type: str
    if (table_desc_sent_id is None) or (table_desc_sent is None): return None

    colors_to_check = [worksheet.cell(s+i, j).fill.fgColor.rgb for i in range(0, 2) for j in range(1, 3)]
    highlight = any([c != _COLOR_WHITE for c in colors_to_check])
    if highlight: return None
    
    block_dict = {
        # position in sheet
        'start_row': s, 'end_row': e, 
        # general table info
        'table_desc_sent_id': table_desc_sent_id, 
        'table_desc_sent': table_desc_sent, 
        
    }
    return block_dict
    

def get_annotated_blocks(worksheet):
    """Get a list of annotated sub-sentence blocks from the given spreadsheet."""
    blocks = []

    # Get basic table shape
    max_row = worksheet.max_row
    # max_column = worksheet.max_column

    # Find the starting row of annotations
    irow = 1
    while irow < max_row:
        if worksheet.cell(irow, 1).value == _TABLE_DESC_SENT_ID: break
        irow += 1
    # if irow == max_row:
    #     return blocks
    
    while irow < max_row:
        erow = irow + 8
        while erow < max_row:
            if worksheet.cell(erow, 1).value == _TABLE_DESC_SENT_ID:
                blocks.append((irow, erow - 1))
                irow = erow
                break
            else:
                erow += 1

        if erow == max_row:
            blocks.append((irow, max_row)) 
            break

    stripped_blocks = [strip_block(block, worksheet) for block in blocks]
    stripped_blocks = [block for block in stripped_blocks if (block is not None)]
    
    block_starts = [start_block(blk, worksheet) for blk in stripped_blocks]
    block_starts = [blk for blk in block_starts if (blk is not None)]
    return block_starts



def refine_block_sample(block_dict):   # currently not functioning
    """Refine the block-dict to ensure each (sub-)field meets requirement."""

    sent_id = block_dict['table_desc_sent_id']
    sent = block_dict['table_desc_sent']
    if not (sent_id and sent): return None

    nsubs = len(block_dict['sub_samples'])
    for idx in range(nsubs):
        if block_dict['sub_samples'][idx]['sub_sent_comp'] is None:
            block_dict['sub_samples'][idx]['sub_sent_comp'] = sent
        if block_dict['sub_samples'][idx]['sub_sent_after'] is None:
            block_dict['sub_samples'][idx]['sub_sent_after'] = block_dict['sub_samples'][idx]['sub_sent_comp']

    return block_dict        

