import json
import requests


def get_model_response(instruction: str, url: str, model: str, input_str: str = "",
                       max_tokens: int = 512) -> str:

    """
    Send an instruction to the model and get the response.
    Args:
        instruction (str): The instruction to send to the model.
        url (str): URL of the model server.
        model (str): Name of the model to use.
        input_str (str): Additional input information for the model.
        max_tokens (int): Maximum number of tokens in the model's response.

    Returns:
        str: The response from the model.
    """
    prompt = generate_prompt(instruction, input_str)
    return request_model(prompt, url, model, max_tokens)


def request_model(prompt: str, url: str, model: str, max_tokens: int = 512) -> str:
    """
        Send a request to the model server and retrieve the response.

        Args:
            prompt (str): The prompt to send to the model.
            url (str): URL of the model server.
            model (str): Name of the model to use.
            max_tokens (int): Maximum number of tokens in the model's response.

        Returns:
            str: The text response from the model.
        """
    headers = {"Content-Type": "application/json"}
    data = {
        "model": model,
        "max_tokens": max_tokens,
        "prompt": prompt,
        "temperature": 0.1,
        "top_p": 0.75,
    }

    try:
        response = requests.post(url, data=json.dumps(data), headers=headers)
        response.raise_for_status()
        response_data = response.json()
        print(f"Model response: {response.text}")
        return response_data["choices"][0]["text"]
    except requests.RequestException as e:
        print(f"Request failed: {e}")
        return ""


def generate_prompt(instruction: str, input_str: str = None) -> str:
    """
    Generate a full prompt for a given instruction and optional input.

    Args:
        instruction (str): The instruction that describes a task.
        input_str (str, optional): Additional input providing context for the task.

    Returns:
        str: A formatted prompt string combining the instruction and input.
    """
    if input_str:
        prompt = ("Below is an instruction that describes a task, "
                  "paired with an input that provides further context. "
                  "Write a response that appropriately completes the request.\n\n"
                  "### Instruction:\n{instruction}\n\n"
                  "### Input:\n{input}\n\n"
                  "### Response:\n")
        return prompt.format(instruction=instruction, input=input_str)
    else:
        # prompt = ("Below is an instruction that describes a task. "
        #           "Write a response that appropriately completes the request.\n\n"
        #           "### Instruction:\n{instruction}\n\n"
        #           "### Response:\n")
        return instruction
