
import torch


def pad_fixed_sequence(seq, padding=1, seq_len=7):
    pad_length = seq_len - len(seq)
    padded_sequence = seq if pad_length == 0 else torch.cat((torch.tensor([padding]*pad_length,
                                                                          device=seq.device), seq))
    return padded_sequence


def generate_incremental_net_input(hypos, num_src=7, src_pad=1, trg_pad=1, distort_action_sequence=False):
    # bsz x src_len x feat_len
    # 'action_seq -1' is because the first action will not be generated by the Agent
    max_action_len = max([len(hypo['action_seq']) for hypo in hypos]) if not distort_action_sequence \
        else max([len(hypo['action_seq']) for hypo in hypos]) - 1

    device = hypos[0]['src_tokens'].device
    final_input = torch.full(
        (len(hypos), max_action_len-1, num_src+1), src_pad
    ).long().to(device)
    final = {
        'src_tokens': [],
        'src_features': [],
        'trg_tokens': [],
        'actions': [],
    }
    
    for i, hypo in enumerate(hypos):
        action_sequence = hypo['action_seq'].copy()
        if distort_action_sequence:
            # Ignore -1 token in the action sequence.
            action_sequence.remove(-1)
        
        sub_input = torch.full((len(action_sequence)-1, num_src+1), src_pad)
        cum_trg = torch.full(
            (sum(action_sequence),), src_pad
        ).long().to(device)
        cum_src = torch.full(
            (len(action_sequence) - sum(action_sequence),), src_pad
        ).long().to(device)

        cum_src_tokens, cum_trg_tokens = [], []
        cum_src_feats = []

        for j in range(1, len(action_sequence)):
            # num read and write at previous time step
            num_write = sum(action_sequence[:j])
            num_read = len(action_sequence[:j]) - num_write

            src_tokens = pad_fixed_sequence(hypo['src_tokens'][num_read-num_src:num_read] if num_read >= num_src
                                            else hypo['src_tokens'][:num_read], padding=src_pad, seq_len=num_src)

            # The last write is EOS. We just need to continue to read more source words.
            # if num_write == sum(hypo['action_seq'][:]):
            #     num_write = num_write-1

            # The last read is EOS token. So it's the same as the lsat trg tokens
            src_size = hypo['src_tokens'].ne(src_pad).sum()
            if (num_read == len(action_sequence[:]) - sum(action_sequence[:])
                and num_read == src_size):
                num_read = num_read-1

            # If the number of generated tokens less than the token we are looking for, then we just send
            # the padding and we expect the Agent generate read action for that time step.
            if num_write >= len(hypo['subsets'][num_read-1]['tokens']):
                trg_tokens = trg_pad
            else:
                trg_tokens = hypo['subsets'][num_read-1]['tokens'][num_write]
            input_elements = torch.cat((src_tokens.type(torch.LongTensor),
                                        torch.tensor(
                                            [trg_tokens],
                                            device=hypos[0]['src_tokens'].device
                                        ).type(torch.LongTensor)))

            sub_input[j-1] = input_elements
            cum_src[num_read - 1] = src_tokens.item()
            cum_trg[num_write] = trg_tokens
            cum_src_tokens.append(cum_src.clone())
            # cum_src_feats.append(hypo['subsets'][num_read - 1]['src_features'].clone())
            cum_trg_tokens.append(cum_trg.clone())

        final_input[i][:sub_input.shape[0]] = sub_input
        final['src_tokens'].append(torch.stack(cum_src_tokens))
        # final['src_features'].append(torch.stack(cum_src_feats).cpu())
        final['trg_tokens'].append(torch.stack(cum_trg_tokens))
        final['actions'].append(action_sequence[1:])

    return final_input, final


def generate_incremental_net_input_2(hypos):
    # bsz x src_len x feat_len
    bsz = len(hypos)
    encoder_embed_dim = hypos[0]['encoder_out'].shape[1]
    decoder_embed_dim = hypos[0]['decoder_out'].shape[1]
    action_lens = [len(hypo['action_seq']) for hypo in hypos]

    # "+1" for attention weight
    feat_len = encoder_embed_dim + decoder_embed_dim + 1

    # 'action_lens -1' is because the first action will not be generated by the Agent
    final_input = torch.zeros(
        bsz, max(action_lens)-1, feat_len,
        dtype=torch.float,
        device=hypos[0]['src_tokens'].device
    )

    for i, hypo in enumerate(hypos):
        sub_input = torch.zeros(len(hypo['action_seq'])-1, feat_len,
                                device=hypos[0]['src_tokens'].device)
        for j in range(1, len(hypo['action_seq'])):
            # num read and write at previous time step
            num_write = sum(hypo['action_seq'][:j])
            num_read = len(hypo['action_seq'][:j]) - num_write

            # The last write is EOS. We just need to continue to read more source words.
            # if num_write == sum(hypo['action_seq'][:]):
            #     num_write = num_write-1

            # The last read is EOS token. So it's the same as the last trg token
            if num_read == len(hypo['action_seq'][:]) - sum(hypo['action_seq'][:]):
                num_read = num_read-1

            src_embedding = hypo['subsets'][num_read-1]['encoder_out'][num_read-1]
            # If the number of generated tokens are less than the token we are looking for, then we just send
            # the padding and we expect the Agent generate read action for that time step.
            if num_write >= hypo['subsets'][num_read-1]['decoder_out'].shape[0]:
                trg_embedding = torch.zeros(decoder_embed_dim)
                attention = 0
            else:
                trg_embedding = hypo['subsets'][num_read-1]['decoder_out'][num_write]
                attention = torch.mean(hypo['subsets'][num_read-1]['attention'][:, num_write])

            sub_input[j-1][:encoder_embed_dim] = src_embedding
            sub_input[j-1][encoder_embed_dim:feat_len-1] = trg_embedding
            sub_input[j-1][-1] = attention
        final_input[i][:sub_input.shape[0]] = sub_input[:].detach().clone()
    return final_input


def prepare_input(hypos, net_input, sample, agt_dict, distort_action_sequence=False):
    agt_pad = agt_dict.pad_index
    device = sample['net_input']['src_tokens'].device

    if distort_action_sequence:
        # Fix ids
        id_idxs = torch.tensor([hypo['id_index'] for hypo in hypos]).to(device=device)
        ids = torch.index_select(sample['id'], dim=0, index=id_idxs)
        sample['id'] = ids.to(device=device)
        sample['nsentences'] = len(ids)

    sample['net_input']['src_tokens'] = net_input
    bsz = net_input.shape[0]
    action_len = net_input.shape[1] if not distort_action_sequence else net_input.shape[1] + 1
    # The first read in the action sequence will not be generated by the Agent
    sample['net_input']['src_lengths'] = torch.tensor([len(hypo['action_seq'][1:]) for hypo in hypos],
                                                      dtype=torch.long, device=device)
    if distort_action_sequence:
        sample['net_input']['src_lengths'] = sample['net_input']['src_lengths'] - 1

    sample['target'] = torch.empty(bsz, action_len, dtype=torch.long, device=device).fill_(agt_pad)
    sample['net_input']['prev_output_tokens'] = torch.empty(
        bsz, action_len, dtype=torch.long, device=device
    ).fill_(agt_pad)
    for i, hypo in enumerate(hypos):
        encoded_action = agt_dict.encode_line(
            " ".join(str(elem) for elem in hypo['action_seq']),
            add_if_not_exist=False,
            append_eos=False
        )
        sample['target'][i][:len(hypo['action_seq'][1:])] = encoded_action[1:]
        sample['net_input']['prev_output_tokens'][i][:len(hypo['action_seq'][:-1])] = encoded_action[:-1]
    return sample


def prepare_simultaneous_input(hypos, sample, task, trg_feat_fn=None):
    agt_dict = task.agent_dictionary
    pad = task.source_dictionary.pad_index
    
    sample_ids = sample['id'].tolist()
    act_lens = torch.tensor([len(hyp['action_seq']) for hyp in hypos])
    del_idxs = (act_lens > 400).nonzero().view(-1)
    if del_idxs.numel() > 0:
        for d in del_idxs.tolist()[::-1]:
            del sample_ids[d]
            del hypos[d]

    if task.cfg.arch in ["agent_lstm", "agent_lstm_big", "agent_lstm_0", "agent_lstm_0_big"]:
        net_input, iq_input = generate_incremental_net_input(
            hypos, num_src=1,
            src_pad=pad,
            trg_pad=pad,
            distort_action_sequence=task.cfg.distort_action_sequence,
        )

        iq_input['ids'] = sample_ids
        sample['id'] = torch.tensor(sample_ids)

        if trg_feat_fn is not None:
            trg_features = []
            for trg_token in iq_input['trg_tokens']:
                feature = trg_feat_fn(trg_token)
                b, t, d = feature.size()
                mask = (trg_token == pad)
                mask = mask.unsqueeze(-1).repeat(1, 1, d)
                padded = torch.zeros(b, 200, d).to(feature)
                padded[:, :t] = feature[:, :200]
                trg_features.append(padded.cpu())
            iq_input['trg_features'] = trg_features
        
        return prepare_input(hypos, net_input, sample, agt_dict,
                             distort_action_sequence=task.cfg.distort_action_sequence), iq_input
    else:
        net_input = generate_incremental_net_input_2(hypos)
        return prepare_input(hypos, net_input, sample, agt_dict), None


def modify_preprocessed_input(sample, task):
    if task.cfg.arch.startswith("agent_lstm_5") or task.cfg.arch.startswith("info"):
        num_prev_tokens = 3  # How many previous tokens should we send in each time
        pad_index = task.agent_dictionary.pad_index
        unk_index = task.agent_dictionary.unk_index

        if unk_index in sample['action_seq']:
            # At least one sample is distorted
            new_action_seq = torch.empty(
                sample['action_seq'].shape[0], sample['action_seq'].shape[1]
            ).fill_(pad_index)    # Use for prev_token
            target = torch.ones_like(sample['target'])  # Use for target
            for index, action_seq in enumerate(sample['action_seq']):
                if unk_index not in action_seq:
                    new_action_seq[index] = action_seq
                    target[index] = action_seq[1:].contiguous()
                else:
                    dist_index = (action_seq == unk_index).nonzero().squeeze()
                    new_action_seq[index][:-1] = action_seq[action_seq != unk_index]
                    target[index] = new_action_seq[index][1:].contiguous()
                    target[index][dist_index-2] = 9 - action_seq[dist_index-1]  # Turns 4 to 5 and vice versa
            padded_sample = torch.empty(
                new_action_seq.shape[0], new_action_seq.shape[1] + num_prev_tokens - 1
            ).fill_(pad_index)
            prev_token = torch.empty(new_action_seq.shape[0], new_action_seq.shape[1] - 1, num_prev_tokens,
                                     device=sample['net_input']['src_tokens'].device)
            padded_sample[:, num_prev_tokens - 1:] = new_action_seq
        else:
            padded_sample = torch.empty(
                sample['action_seq'].shape[0], sample['action_seq'].shape[1] + num_prev_tokens - 1
            ).fill_(pad_index)
            prev_token = torch.empty(sample['action_seq'].shape[0], sample['action_seq'].shape[1] - 1, num_prev_tokens,
                                     device=sample['net_input']['src_tokens'].device)
            target = sample['action_seq'][:, 1:].contiguous()
            padded_sample[:, num_prev_tokens-1:] = sample['action_seq']

        for i in range(num_prev_tokens):
            prev_token[:, :, i] = padded_sample[:, i:i-num_prev_tokens]
    else:
        prev_token = sample['action_seq'][:, :-1]
        target = sample['action_seq'][:, 1:].contiguous()
    net_input = {
        'src_tokens': torch.cat((sample['net_input']['src_tokens'].unsqueeze(2),
                                 sample['target'].unsqueeze(2)), dim=2),
        'src_lengths': sample['net_input']['src_lengths'],
        'prev_output_tokens': prev_token.type(torch.LongTensor).to(device=sample['net_input']['src_tokens'].device)
    }
    new_sample = {
        'id': sample['id'],
        'nsentences': sample['nsentences'],
        'ntokens': sample['ntokens'],
        'net_input': net_input,
        'target': target
    }
    return new_sample


def infer_input_features(hypos, previous_actions, all_features=True):
    bsz = len(hypos)
    pad = 1
    device = hypos[0][0]['tokens'].device
    if all_features:
        encoder_embed_dim = hypos[0][0]['encoder_out'].shape[1]
        decoder_embed_dim = hypos[0][0]['decoder_out'].shape[1]
        feat_len = 1 + encoder_embed_dim + decoder_embed_dim
    else:
        feat_len = 2

    num_writes = torch.sum(torch.eq(previous_actions, 5), dim=1)
    num_reads = torch.sum(torch.eq(previous_actions, 4), dim=1)

    final_features = torch.zeros(bsz, feat_len, dtype=torch.float, device=device)
    src_tokens = torch.zeros(bsz, dtype=torch.int, device=device)
    trg_tokens = torch.zeros(bsz, dtype=torch.int, device=device)

    for i, hypo in enumerate(hypos):
        num_read = int(num_reads[i])
        num_write = int(num_writes[i])
        send_padding = True if num_write > 1 and num_write >= len(hypo[num_read - 1]['tokens']) else False

        # We will use src and trg tokens for checking stopping criteria in the Agent.
        src_tokens[i] = hypo[-1]['src_tokens'][num_read - 1]
        trg_tokens[i] = hypo[num_read-1]['tokens'][num_write] if not send_padding else pad

        if all_features:
            src_embedding = hypo[num_read - 1]['encoder_out'][num_read - 1]
            trg_embedding = hypo[num_read - 1]['decoder_out'][num_write] \
                if not send_padding else torch.zeros(decoder_embed_dim)
            attention = torch.mean(hypo[num_read - 1]['attention'][:, num_write]) \
                if not send_padding else 0
            final_features[i][:encoder_embed_dim] = src_embedding
            final_features[i][encoder_embed_dim:feat_len-1] = trg_embedding
            final_features[i][-1] = attention
        else:
            final_features[i] = torch.cat((torch.tensor([src_tokens[i]], device=device).type(torch.LongTensor),
                                           torch.tensor([trg_tokens[i]], device=device).type(torch.LongTensor)))
    final_dict = {
        'src_tokens': src_tokens,
        'trg_tokens': trg_tokens,
        'input_features': final_features
    }
    return final_dict


def info_input_features(hypos, previous_actions, nmt_model, all_features=True):
    bsz = len(hypos)
    pad = 1
    device = hypos[0][0]['tokens'].device
    if all_features:
        encoder_embed_dim = hypos[0][0]['encoder_out'].shape[1]
        decoder_embed_dim = hypos[0][0]['decoder_out'].shape[1]
        feat_len = 1 + encoder_embed_dim + decoder_embed_dim
    else:
        feat_len = 2

    num_writes = torch.sum(torch.eq(previous_actions, 5), dim=1)
    num_reads = torch.sum(torch.eq(previous_actions, 4), dim=1)

    final_features = torch.zeros(bsz, feat_len, dtype=torch.float, device=device)
    ls, lt = num_reads.max().item(), (num_writes.max().item() + 1)
    src_tokens = torch.ones((bsz, ls), dtype=torch.int, device=device)
    trg_tokens = torch.ones((bsz, lt), dtype=torch.int, device=device)

    for i, hypo in enumerate(hypos):
        num_read = int(num_reads[i])
        num_write = int(num_writes[i])
        send_padding = True if num_write > 1 and num_write >= len(hypo[num_read - 1]['tokens']) else False

        # We will use src and trg tokens for checking stopping criteria in the Agent.
        src_tokens[i, :num_read] = hypo[-1]['src_tokens'][:num_read]
        if not send_padding:
            trg_tokens[i, :num_write + 1] = hypo[num_read-1]['tokens'][:num_write + 1]

        final_features[i] = torch.cat((torch.tensor([src_tokens[i, num_read - 1]], device=device).type(torch.LongTensor),
                                        torch.tensor([trg_tokens[i, num_write]], device=device).type(torch.LongTensor)))
    
    src_features = nmt_model.encoder(src_tokens)['encoder_out'][0]
    src_features = src_features.transpose(1,0).contiguous()
    trg_features = nmt_model.decoder(
        encoder_out=None,
        features_only=True,
        prev_output_tokens=trg_tokens,
    )[0]

    final_dict = {
        'src_tokens': final_features[:, 0].long(),
        'trg_tokens': final_features[:, 1].long(),
        'input_features': final_features,
        'net_input': {
            'src_features': src_features,
            'trg_features': trg_features,
            'src_tokens': src_tokens,
            'trg_tokens': trg_tokens,
        }
    }
    return final_dict

def cal_info(net_output, padding_mask):
    src_info = net_output[0]['src_info']
    trg_info = net_output[0]['trg_info']

    src_pad = padding_mask['src_pad'].unsqueeze(-1)
    trg_pad = padding_mask['trg_pad'].unsqueeze(-1)
    src_info.masked_fill_(src_pad, 0.)
    trg_info.masked_fill_(trg_pad, 0.)

    src_info_bias = ~(src_info == 0)
    trg_info_bias = ~(trg_info == 0)

    src_info = torch.nn.functional.softplus(src_info) #+ src_info_bias
    trg_info = torch.nn.functional.softplus(trg_info) #+ trg_info_bias
    
    src_info = src_info.sum(1).squeeze()
    trg_info = trg_info.sum(1).squeeze()
    valid_info = src_info - trg_info
    
    return valid_info