import { Meteor } from 'meteor/meteor';
import { Texts, Annotations, Assignments, Tutorials } from '/imports/api/documents';

Meteor.startup(() => {
  if (Texts.find().count() === 0) {
    // Tutorial
    const tutorials = [
      {
        ui: 'mentions',
        name: 'mentions-t0',
        tasks: [
          { 
            context: "In this task, you will be identifying references to things in text. This tutorial will use examples to define the task and show you how to do it.",
            progress: "Begin",
            text: [],
            anno: [],
            init: [],
          },
          { 
            context: "This sentence contains a reference to the company 'Google' and to the state 'California'. Click on the word 'Google' to label it as a reference. Do the same for 'California'. If you make a mistake, remove the label by holding alt/option and clicking anywhere inside the reference.",
            progress: "Check",
            text: ["Google", "is", "located", "in", "California", "."],
            anno: [[[0, 0]], [[4, 4]]], 
            init: [],
          },
          { 
            context: "For this sentence we have already provided three labels, two of which are incorrect. To remove each label, hold alt/option and click anywhere inside the reference.",
            progress: "Check",
            text: ["It", "is", "really", "big", "."],
            anno: [[[0, 0]]], 
            init: [[[0, 0]], [[1, 1]], [[3, 3]]],
          },
          {
            context: "This example contains multi-word references (e.g. 'The company'). To label them, click and drag. Also remember to label the other references in the sentence. To decide whether to label one word or several, consider whether they could be replaced by a pronoun. For example, our example was 'The company' rather than just 'company' because 'It announced ...' makes more sense than 'The it announced ...'",
            progress: "Check",
            text: ["The", "company", "announced", "it", "was", "providing", "free", "books", "to", "employees", "."],
            anno: [[[0, 1]], [[3, 3]], [[6, 7]], [[9, 9]]],
            init: [],
          },
          { 
            context: "This example contains references that overlap: 'Google' refers to a company, while 'Google 's free book program' refers to a service.",
            progress: "Check",
            text: ["Google", "'s", "free", "book", "program", "is", "an", "unusual", "choice", "for", "a", "tech", "company", "."],
            anno: [[[0, 0]], [[0, 4]], [[6, 8]], [[10, 12]]],
            init: [],
          }
        ]
      }
    ];

    tutorials.forEach(tutorial => {
      const tasks = tutorial.tasks.map((example, eid) => {
        // Convert into structures for DB
        const tokens = example.text.map((token, tid) => {
          return {_id: tid, text: token};
        });
        var labels = new Set();
        if (example.labels != undefined) {
          labels = example.labels;
        }
        const annotations = example.anno.map((pairs, mid) => {
          return {
            _id: "mention-"+ tutorial.name +"-"+ String(eid) +"-"+ String(mid),
            mentions: pairs.map(pair => {
              if (pair.length > 2 && example.labels === undefined) {
                labels.add(pair[2]);
              }
              return { start: pair[0], end: pair[1], label: (pair.length > 2 ? pair[2] : "") }
            }),
            color: "#7570b3",
          };
        });
        const init_annotations = example.init.map((pairs, mid) => {
          return {
            _id: "i-mention-"+ tutorial.name +"-"+ String(eid) +"-"+ String(mid),
            mentions: pairs.map(pair => {
              return { start: pair[0], end: pair[1], label: (pair.length > 2 ? pair[2] : "") }
            }),
            color: "#7570b3",
          };
        });
        const labelsArray = Array.from(labels);
        const focus = example.focusMention === undefined ? null : {
          start: example.focusMention[0],
          end: example.focusMention[1],
          label: (example.focusMention.length > 2 ? example.focusMention[2] : "")};

        // Insert into DB
        const text_id = Texts.insert({
          _id: "tutorial-text" + tutorial.name +"-"+ String(eid),
          tokens: tokens,
        });
        const anno_id = Annotations.insert({
          _id: "tutorial-ann" + tutorial.name +"-"+ String(eid),
          user: "server",
          text_id: text_id,
          clusters: annotations,
          focus_mention: focus,
          labels: labelsArray,
          checks: [],
          createdAt: new Date(),
        });
        const init_anno_id = Annotations.insert({
          _id: "tutorial-i-ann" + tutorial.name +"-"+ String(eid),
          user: "server",
          text_id: text_id,
          clusters: init_annotations,
          focus_mention: focus,
          labels: labelsArray,
          checks: [],
          createdAt: new Date(),
        });

        return { 
          text_id: text_id,
          anno_id: anno_id,
          init_anno_id: init_anno_id,
          context: example.context,
          progress: example.progress,
        };
      });

      // Insert tutorial
      Tutorials.insert({
        _id: tutorial.name,
        ui: tutorial.ui,
        tasks: tasks,
      });
    });

    // Insert assignments with username 'tutorial'

    // Sample documents

    const examples = [
      { // /wsj_1024.coref
        text: ["Eastman", "Kodak", "Co.", "of", "Rochester", ",", "N.Y.", ",",
          "said", "its", "Lehn", "&", "Fink", "Products", "subsidiary", "was",
          "restructured", "to", "form", "two", "operating", "groups", ",",
          "one", "for", "household", "products", "such", "as", "cleaners",
          "and", "disinfectants", "and", "the", "other", "for", "do", "-",
          "it", "-", "yourself", "products", "such", "as", "wood", "finishes",
          "and", "stains", ".", "The", "Montvale", ",", "N.J.", ",", "unit",
          "said", "the", "new", "operating", "structure", "``", "creates", "a",
          "more", "focused", "and", "responsive", "organization", "geared",
          "to", "effectively", "managing", "the", "size", "and", "scope", "of",
          "the", "unit", "'s", "current", "business", ".", "''", "The",
          "consumer", "brands", "unit", "was", "absorbed", "by", "the",
          "photographic", ",", "pharmaceutical", "and", "chemical", "concern",
          "last", "year", "when", "it", "acquired", "Sterling", "Drug", "Inc",
          ".", "In", "a", "related", "matter", ",", "Peter", "Black", ",",
          "president", "of", "consumer", "brand", "Minwax", ",", "was",
          "named", "group", "vice", "president", "for", "the", "household",
          "products", "operating", "group", ".", "Kenneth", "M.", "Evans", ",",
          "president", "of", "Thompson", "&", "Formby", "brand", ",", "was",
          "named", "group", "vice", "president", "of", "the", "do", "-", "it",
          "-", "yourself", "operating", "group", "."]
      },
      { // /wsj_1024.coref
        text: ["Information", "International", "Inc.", "said", "it", "was",
          "sued", "by", "a", "buyer", "of", "its", "computerized", "newspaper",
          "-", "publishing", "system", ",", "alleging", "that", "the",
          "company", "failed", "to", "correct", "deficiencies", "in", "the",
          "system", ".", "A", "spokesman", "for", "Information",
          "International", "said", "the", "lawsuit", "by", "two", "units",
          "of", "Morris", "Communications", "Corp.", "seeks", "restitution",
          "of", "the", "system", "'s", "about", "$", "3", "million",
          "purchase", "price", "and", "cancellation", "of", "a", "software",
          "license", "provided", "by", "the", "Morris", "units", "to",
          "Information", "International", "for", "alleged", "failure", "to",
          "pay", "royalties", ".", "Information", "International", "said",
          "it", "believes", "that", "the", "complaints", ",", "filed", "in",
          "federal", "court", "in", "Georgia", ",", "are", "without", "merit",
          ".", "Closely", "held", "Morris", "Communications", "is", "based",
          "in", "Augusta", ",", "Ga", ".", "The", "units", "that", "filed",
          "the", "suit", "are", "Southeastern", "Newspapers", "Corp.", "and",
          "Florida", "Publishing", "Co", "."]
      },
      { // Alice in Wonderland
        text: [
          "Alice", "was", "beginning", "to", "get", "very", "tired", "of",
          "sitting", "by", "her", "sister", "on", "the", "bank", ",", "and",
          "of", "having", "nothing", "to", "do", ":", "once", "or", "twice",
          "she", "had", "peeped", "into", "the", "book", "her", "sister",
          "was", "reading", ",", "but", "it", "had", "no", "pictures", "or",
          "conversations", "in", "it", ",", "'", "and", "what", "is", "the",
          "use", "of", "a", "book", ",", "'", "thought", "Alice", "'",
          "without", "pictures", "or", "conversations", "?", "'", "So", "she",
          "was", "considering", "in", "her", "own", "mind", "(", "as", "well",
          "as", "she", "could", ",", "for", "the", "hot", "day", "made", "her",
          "feel", "very", "sleepy", "and", "stupid", ")", ",", "whether",
          "the", "pleasure", "of", "making", "a", "daisy-chain", "would", "be",
          "worth", "the", "trouble", "of", "getting", "up", "and", "picking",
          "the", "daisies", ",", "when", "suddenly", "a", "White", "Rabbit",
          "with", "pink", "eyes", "ran", "close", "by", "her", "."]
      },
    ]

    examples.forEach((example, index) => {
      const tokens = example.text.map((token, tid) => {
        return {_id: tid, text: token};
      });

      Texts.insert({
        _id: "example-text" + String(index),
        tokens: tokens,
      });
    });


    Texts.find( { _id: { $regex: /^example-text/ } } ).fetch().forEach(text => {
      Annotations.insert({
        text_id: text._id,
        user: "server",
        clusters: [],
        labels: ["LOC", "PER", "ORG", "MISC"],
        checks: [],
        focus_mention: null,
        createdAt: new Date(),
      });
    });

    Annotations.find().fetch().forEach(anno => {
      if (anno.text_id.startsWith("example")) {
        ["full"].forEach(ui => {
          Assignments.insert({
            group: "example" +"_"+ ui,
            tutorial: "mentions-t0",
            count: 3,
            text_id: anno.text_id,
            ann_id: anno._id,
            ui: ui,
            createdAt: new Date(),
          });
        });
      }
    });
  }
});
