# Libraries

import pandas as pd
import torch

# Preliminaries
from sklearn.model_selection import train_test_split
from sklearn.metrics import accuracy_score, precision_recall_fscore_support
from torchtext.data import Field, Dataset, BucketIterator, Example
# word embeddings
import fasttext.util

import warnings
warnings.filterwarnings("ignore")

# Code for baseline model BiLSTM from Schulz et al., 2019

destination_folder = "./model_output"
# cc.de.300.bin
fasttext_cc_de_300 = "/path/to/fasttext/embeddings"

# Models

import torch.nn as nn
from torch.nn.utils.rnn import pack_padded_sequence, pad_packed_sequence

device = torch.device('cuda:0' if torch.cuda.is_available() else 'cpu')
print(device)

label_map = {
    'Unrelated': 0,
    'Comment': 1,
    'Support': 2,
    'Refute': 3
}
inputpath = "../data/expertdata_resolved.tsv"
# Fields
label_column = "gold"
df = pd.read_csv(inputpath, sep="\t")
num_labels = len(df[label_column].unique())
df["label"] = df["label"].map(lambda x: label_map[x])
label_field = Field(sequential=False, use_vocab=False, batch_first=True, dtype=torch.int64)
text_field = Field(lower=False, include_lengths=True, batch_first=True)
fields = {'label': ('label', label_field), 'full_text': ('text', text_field)}
dataset_fields = [('label', label_field), ('text', text_field)]


# build embedding matrix
all_examples = [Example.fromdict(x, fields=fields) for x in df.to_dict(orient='records')]
all_dataset = Dataset(examples=all_examples, fields=dataset_fields)
text_field.build_vocab(all_dataset, min_freq=1)
fasttext_model = fasttext.load_model(fasttext_cc_de_300)
embeddings = torch.Tensor(len(text_field.vocab), 300)
for i, token in enumerate(text_field.vocab.itos):
    wv_index = text_field.vocab.stoi.get(token, None)
    embeddings[i] = embeddings[wv_index]

# Iterators


# variables
num_epochs = 50
lr = 0.005
import torch.optim as optim
print('epochs: {}, lr: {}'.format(num_epochs, lr))

class LSTM(nn.Module):

    def __init__(self, embeddings, num_labels=4, dimension=100):
        super(LSTM, self).__init__()

        self.embedding = nn.Embedding.from_pretrained(embeddings=embeddings, freeze=False)
        self.dimension = dimension
        self.lstm = nn.LSTM(input_size=300,
                            hidden_size=dimension,
                            num_layers=1,
                            batch_first=True,
                            bidirectional=True)
        self.drop = nn.Dropout(p=0.25)

        self.fc = nn.Linear(2*dimension, num_labels)

    def forward(self, text, text_len):

        text_emb = self.embedding(text)

        packed_input = pack_padded_sequence(text_emb, text_len.cpu(), batch_first=True, enforce_sorted=False)
        packed_output, _ = self.lstm(packed_input)
        output, _ = pad_packed_sequence(packed_output, batch_first=True)

        out_forward = output[range(len(output)), text_len - 1, :self.dimension]
        out_reverse = output[:, 0, self.dimension:]
        out_reduced = torch.cat((out_forward, out_reverse), 1)
        text_fea = self.drop(out_reduced)

        text_fea = self.fc(text_fea)
        text_fea = torch.squeeze(text_fea, 1)
        text_out = torch.sigmoid(text_fea)

        return text_out

# Save and Load Functions

def save_checkpoint(save_path, model, optimizer, valid_loss):

    if save_path == None:
        return

    state_dict = {'model_state_dict': model.state_dict(),
                  'optimizer_state_dict': optimizer.state_dict(),
                  'valid_loss': valid_loss}

    torch.save(state_dict, save_path)
    #print(f'Model saved to ==> {save_path}')


def load_checkpoint(load_path, model, optimizer):

    if load_path==None:
        return

    state_dict = torch.load(load_path, map_location=device)
    print(f'Model loaded from <== {load_path}')

    model.load_state_dict(state_dict['model_state_dict'])
    optimizer.load_state_dict(state_dict['optimizer_state_dict'])

    return state_dict['valid_loss']


def save_metrics(save_path, train_loss_list, valid_loss_list, global_steps_list):

    if save_path == None:
        return

    state_dict = {'train_loss_list': train_loss_list,
                  'valid_loss_list': valid_loss_list,
                  'global_steps_list': global_steps_list}

    torch.save(state_dict, save_path)
    #print(f'Model saved to ==> {save_path}')


def load_metrics(load_path):

    if load_path==None:
        return

    state_dict = torch.load(load_path, map_location=device)
    print(f'Model loaded from <== {load_path}')

    return state_dict['train_loss_list'], state_dict['valid_loss_list'], state_dict['global_steps_list']

# Training Function

def train(model,
          optimizer,
          train_loader,
          valid_loader,
          criterion = nn.CrossEntropyLoss(),
          num_epochs = 10,
          eval_every = 10000,
#len(train_iter) // 10,
          file_path = destination_folder,
          best_valid_loss = float("Inf")
          ):

    # initialize running values
    running_loss = 0.0
    valid_running_loss = 0.0
    global_step = 0
    train_loss_list = []
    valid_loss_list = []
    global_steps_list = []
    best_f1 = 0.0
    best_acc = 0.0
    best_epoch = None

    # training loop
    model.train()
    for epoch in range(num_epochs):
        for (labels, (text, text_len)), _ in train_loader:
            labels = labels.to(device)
            text = text.to(device)
            text_len = text_len.to(device)
            output = model(text, text_len)
#            output = torch.argmax(output,dim=1)
            loss = criterion(output, labels)
            optimizer.zero_grad()
            loss.backward()
            optimizer.step()

            # update running values
            running_loss += loss.item()
            global_step += 1

            # evaluation step
            if global_step % eval_every == 0:
                model.eval()
                with torch.no_grad():
                    # validation loop
                    predicted_labels = []
                    true_labels = []
                    for (labels, (text, text_len)), _ in valid_loader:
                        labels = labels.to(device)
                        text = text.to(device)
                        text_len = text_len.to(device)
                        output = model(text, text_len)
                        predicted_labels.extend(torch.argmax(output, dim=1).cpu())
                        true_labels.extend(labels.cpu())
                        loss = criterion(output, labels)
                        valid_running_loss += loss.item()

                # evaluation
                precision, recall, f1, _ = precision_recall_fscore_support(true_labels, predicted_labels, average='macro')
                accuracy = accuracy_score(true_labels, predicted_labels)
                average_train_loss = running_loss / eval_every
                average_valid_loss = valid_running_loss / len(valid_loader)
                train_loss_list.append(average_train_loss)
                valid_loss_list.append(average_valid_loss)
                global_steps_list.append(global_step)

                # resetting running values
                running_loss = 0.0
                valid_running_loss = 0.0
                model.train()

                # print progress
                print('Epoch [{}/{}], Step [{}/{}], Train Loss: {:.4f}, Valid Loss: {:.4f}, F1: {:.4f}, Acc: {:.4f}'
                      .format(epoch+1, num_epochs, global_step, num_epochs*len(train_loader),
                              average_train_loss, average_valid_loss, f1, accuracy))

                # checkpoint
                if best_valid_loss > average_valid_loss:
                    best_valid_loss = average_valid_loss
                    save_checkpoint(file_path + '/model.pt', model, optimizer, best_valid_loss)
                    save_metrics(file_path + '/metrics.pt', train_loss_list, valid_loss_list, global_steps_list)


        model.eval()
        with torch.no_grad():
            # validation loop
            predicted_labels = []
            true_labels = []
            for (labels, (text, text_len)), _ in valid_loader:
                labels = labels.to(device)
                text = text.to(device)
                text_len = text_len.to(device)
                output = model(text, text_len)
                predicted_labels.extend(torch.argmax(output, dim=1).cpu())
                true_labels.extend(labels.cpu())
                loss = criterion(output, labels)
                valid_running_loss += loss.item()

        # evaluation
        precision, recall, f1, _ = precision_recall_fscore_support(true_labels, predicted_labels, average='macro')
        accuracy = accuracy_score(true_labels, predicted_labels)
        if f1 > best_f1:
            best_f1 = f1
            best_acc = accuracy
            best_epoch = epoch
        model.train()

    save_metrics(file_path + '/metrics.pt', train_loss_list, valid_loss_list, global_steps_list)
    #print('Finished Training!')
    return best_epoch, best_f1, best_acc

seeds = [2020, 2021, 2022, 2023, 2024, 2025, 2026, 2027, 2028, 2029]
for current_seed in seeds:
    df_train, df_test = train_test_split(df, test_size=0.20, train_size=0.80, shuffle=True, stratify=df[[label_column]],
                                        random_state=current_seed)


    train_examples = [Example.fromdict(x, fields=fields) for x in df_train.to_dict(orient='records')]
    test_examples = [Example.fromdict(x, fields=fields) for x in df_test.to_dict(orient='records')]
    train_dataset = Dataset(examples=train_examples, fields=dataset_fields)
    test_dataset = Dataset(examples=test_examples, fields=dataset_fields)

    train_iter = BucketIterator(train_dataset, batch_size=8, sort_key=lambda x: len(x.text),
                                device=device, sort=True, sort_within_batch=True)
    test_iter = BucketIterator(test_dataset, batch_size=8, sort_key=lambda x: len(x.text),
                                device=device, sort=True, sort_within_batch=True)

    model = LSTM(embeddings=embeddings).to(device)
    optimizer = optim.AdamW(model.parameters(), lr=lr)

    epoch, f1, acc = train(model, optimizer, train_iter, test_iter, num_epochs=num_epochs)
    print('seed: {}, epoch: {}, f1: {:.4f}, acc: {:.2f}'.format(current_seed, epoch, f1, acc))