import torch.nn as nn
import torch
from transformers import AutoModel, AutoTokenizer
import numpy as np
import pyhocon
import os
from inspect import getfullargspec

# relative path of config file


def init_weights(m):
    if isinstance(m, nn.Linear):
        nn.init.xavier_uniform_(m.weight)
        nn.init.uniform_(m.bias)


class Causal_Intervention_Scorer(nn.Module):
    def __init__(self, is_training=True, long=True, model_name='allenai/longformer-base-4096',
                 linear_weights=None, joint_model = True, probing_weights = None, causal_weights = None):
        super(Causal_Intervention_Scorer, self).__init__()
        self.tokenizer = AutoTokenizer.from_pretrained(model_name)
        self.long = long
        self.joint_model = joint_model
        if is_training:
            self.tokenizer.add_tokens(['<m>', '</m>'], special_tokens=True)
            self.tokenizer.add_tokens(['<doc-s>', '</doc-s>'], special_tokens=True)
            self.tokenizer.add_tokens(['<g>'], special_tokens=True)
            self.model = AutoModel.from_pretrained(model_name)
            self.model.resize_token_embeddings(len(self.tokenizer))
        else:
            self.model = AutoModel.from_pretrained(model_name)

        self.start_id = self.tokenizer.encode('<m>', add_special_tokens=False)[0]
        self.end_id = self.tokenizer.encode('</m>', add_special_tokens=False)[0]

        self.hidden_size = self.model.config.hidden_size

        self.linear = nn.Sequential(
            nn.Linear(self.hidden_size * 4, self.hidden_size),
            nn.Tanh(),
            nn.Linear(self.hidden_size, 128),
            nn.Tanh(),
            nn.Linear(128, 1),
            nn.Sigmoid()
          
        )
        self.causal_intevention_layer = nn.Sequential(
            
            nn.Linear(self.hidden_size, 128),
            nn.Tanh(),
            nn.Linear(128, 1),
        
        )

        self.probing_intevention_layer = nn.Sequential(
            
            nn.Linear(self.hidden_size, 128),
            nn.Tanh(),
            nn.Linear(128, 1),
        
        )



        if linear_weights is None:
            self.linear.apply(init_weights)
            if self.joint_model:
                self.causal_intevention_layer.apply(init_weights)
                self.probing_intevention_layer.apply(init_weights)
        else:
            self.linear.load_state_dict(linear_weights)
            if self.joint_model:
                self.causal_intevention_layer.load_state_dict(causal_weights)
                self.probing_intevention_layer.load_state_dict(probing_weights)

    def generate_cls_arg_vectors(self, input_ids, attention_mask, position_ids,
                                 global_attention_mask, arg1, arg2):
        arg_names = set(getfullargspec(self.model).args)

        if self.long:
            output = self.model(input_ids,
                                position_ids=position_ids,
                                attention_mask=attention_mask,
                                global_attention_mask=None)
        else:
            output = self.model(input_ids,
                                attention_mask=attention_mask)

        last_hidden_states = output.last_hidden_state
        cls_vector = output.pooler_output

        arg1_vec = None
        if arg1 is not None:
            arg1_vec = (last_hidden_states * arg1   .unsqueeze(-1)).sum(1)
        arg2_vec = None
        if arg2 is not None:
            arg2_vec = (last_hidden_states * arg2.unsqueeze(-1)).sum(1)
        
        #arg1_vec = torch.mean(last_hidden_states, dim=1) 
        #arg2_vec = torch.mean(last_hidden_states, dim=1)
        
        return cls_vector, arg1_vec, arg2_vec

    def generate_model_output(self, input_ids, attention_mask, position_ids,
                              global_attention_mask, arg1, arg2, cosine_sim=False):
        cls_vector, arg1_vec, arg2_vec = self.generate_cls_arg_vectors(input_ids, attention_mask, position_ids,
                                                                       global_attention_mask, arg1, arg2)
        if(cosine_sim):
            cos = nn.CosineSimilarity(dim=1, eps=1e-6)
            cosine_sim = cos(arg1_vec,  arg2_vec)
            print("getting cosine similarities")
            return cosine_sim
        
        return torch.cat([cls_vector, arg1_vec, arg2_vec, arg1_vec * arg2_vec], dim=1), arg1_vec, arg2_vec

    def frozen_forward(self, input_):
        return self.linear(input_)

    def forward(self, input_ids, attention_mask=None, position_ids=None,
                global_attention_mask=None, arg1=None, arg2=None, lm_only=False, pre_lm_out=False, cosine_sim=False, joint_model = None):

        # attention_mask[global_attention_mask == 1] = 2
        joint_model = self.joint_model
        if pre_lm_out:
            return self.linear(input_ids)

        lm_output, arg1_vec, arg2_vec  = self.generate_model_output(input_ids, attention_mask=attention_mask,
                                               global_attention_mask=global_attention_mask,
                                               position_ids=position_ids,
                                               arg1=arg1, arg2=arg2,cosine_sim=False)
        if lm_only:
            return lm_output
        if cosine_sim:
            return lm_output
        if joint_model:
            return self.linear(lm_output), self.causal_intevention_layer(arg1_vec), self.probing_intevention_layer(arg2_vec)
        
        return self.linear(lm_output)
