# Origin of the code: https://github.com/shuishida/LangProp/blob/main/src/carla_tools/sensors.py

from dataclasses import dataclass, field
from typing import Union, Optional
import numpy as np


@dataclass
class SensorConfig:
    x: float
    y: float
    z: float


@dataclass
class CameraConfig(SensorConfig):
    width: int
    height: int
    fov: Optional[Union[float, int]] = None
    focal_length: Optional[float] = None
    intrinsics: np.ndarray = field(init=False)

    def __post_init__(self):
        assert not (self.fov and self.focal_length), "fov and focal length cannot be declared at the same time"
        if self.fov:
            self.focal_length = self.width / (2.0 * np.tan(np.radians(self.fov / 2)))
        if self.focal_length:
            self.fov = np.degrees(np.arctan(self.width / self.focal_length / 2.0)) * 2
        self.intrinsics = self.get_intrinsics()

    def get_intrinsics(self):
        """
        returns the calibration matrix K for the given sensor
        """
        return np.array([
            [self.focal_length, 0, self.width / 2.0],
            [0, self.focal_length, self.height / 2.0],
            [0, 0, 1]
        ])


def get_sensors(
        camera_config: CameraConfig,
        lidar_config: SensorConfig = None,
        topdown_config: CameraConfig = None,
        use_rgb: bool = True,
):
    """
    sensor.camera.rgb — Regular camera that captures images.
    sensor.lidar.ray_cast — Velodyne 64 LIDAR.
    sensor.other.radar — Long-range RADAR (up to 100 meters).
    sensor.other.gnss — GPS sensor returning geo location data.
    sensor.other.imu — 6-axis Inertial Measurement Unit.
    sensor.opendrive_map — Pseudosensor that exposes the HD map in OpenDRIVE format parsed as a string.
    sensor.speedometer — Pseudosensor that provides an approximation of your linear velocity.

     ^ z
     |
     |
     | . x
     |/
     +-------> y


    """

    # Note that CARLA uses the Unreal Engine coordinate system, which is: x-front, y-right, z-up
    sensors = [
        {
            'type': 'sensor.other.imu',
            'x': 0.0, 'y': 0.0, 'z': 0.0, 'roll': 0.0, 'pitch': 0.0, 'yaw': 0.0,
            "sensor_tick": 0.05,
            'id': 'imu'
        },
        {
            'type': 'sensor.other.gnss',
            'x': 0.0, 'y': 0.0, 'z': 0.0, 'roll': 0.0, 'pitch': 0.0, 'yaw': 0.0,
            "sensor_tick": 0.01,
            'id': 'gps'
        },
        {
            'type': 'sensor.speedometer',
            "reading_frequency": 20,
            'id': 'speed'
        }
    ]
    if use_rgb:
        sensors += [
            {
                "type": "sensor.camera.rgb",
                "x": camera_config.x,
                "y": camera_config.y,
                "z": camera_config.z,
                "roll": 0.0,
                "pitch": 0.0,
                "yaw": 0.0,
                "width": camera_config.width,
                "height": camera_config.height,
                "fov": camera_config.fov,
                "id": "rgb_front",
            },
            {
                "type": "sensor.camera.rgb",
                "x": -camera_config.x,
                "y": camera_config.y,
                "z": camera_config.z,
                "roll": 0.0,
                "pitch": 0.0,
                "yaw": 180.0,
                "width": camera_config.width,
                "height": camera_config.height,
                "fov": camera_config.fov,
                "id": "rgb_rear",
            },
            {
                "type": "sensor.camera.rgb",
                "x": camera_config.x,
                "y": camera_config.y,
                "z": camera_config.z,
                "roll": 0.0,
                "pitch": 0.0,
                "yaw": -60.0,
                "width": camera_config.width,
                "height": camera_config.height,
                "fov": camera_config.fov,
                "id": "rgb_left",
            },
            {
                "type": "sensor.camera.rgb",
                "x": camera_config.x,
                "y": camera_config.y,
                "z": camera_config.z,
                "roll": 0.0,
                "pitch": 0.0,
                "yaw": 60.0,
                "width": camera_config.width,
                "height": camera_config.height,
                "fov": camera_config.fov,
                "id": "rgb_right",
            },
        ]
    if lidar_config:
        sensors += [
            {
                "type": "sensor.lidar.ray_cast",
                "x": lidar_config.x,
                "y": lidar_config.y,
                "z": lidar_config.z,
                "roll": 0.0,
                "pitch": 0.0,
                "yaw": -90.0,
                "id": "lidar",
            },
        ]
    if topdown_config:
        sensors += [
            {
                "type": "sensor.camera.semantic_segmentation",
                "x": topdown_config.x,
                "y": topdown_config.y,
                "z": topdown_config.z,
                "roll": 0.0,
                "pitch": -90.0,
                "yaw": 0.0,
                "width": topdown_config.width,
                "height": topdown_config.height,
                "fov": topdown_config.fov,
                "id": "topdown",
            }
        ]

    return sensors
