package ar.edu.famaf.nlp.webnlg2016;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.PrintWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.Map;
import java.util.HashMap;
import java.util.concurrent.atomic.AtomicInteger;

import org.openrdf.model.Model;
import org.openrdf.model.URI;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.ValueFactory;
import org.openrdf.repository.Repository;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.repository.RepositoryResult;
import org.openrdf.repository.sail.SailRepository;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.RDFParseException;
import org.openrdf.rio.Rio;
import org.openrdf.rio.UnsupportedRDFormatException;
import org.openrdf.sail.memory.MemoryStore;

import ar.edu.famaf.nlp.alusivo.DaleReiterAlgorithm;
import ar.edu.famaf.nlp.alusivo.GardentAlgorithm;
import ar.edu.famaf.nlp.alusivo.GraphAlgorithm;
import ar.edu.famaf.nlp.alusivo.ReferringExpression;
import ar.edu.famaf.nlp.alusivo.ReferringExpression.Predicate;
import ar.edu.famaf.nlp.alusivo.ReferringExpressionAlgorithm;
import ar.edu.famaf.nlp.alusivo.Resolver;
import ar.edu.famaf.nlp.alusivo.TypePriorities;

public class App {

    private static class Data {
        RepositoryConnection connNew;
        RepositoryConnection connOld;

        double[] diceAccum;
        int[] diceCount;
        int[] solvedWithType;
        int[] executionError;
        int[] inclusionErrors;
        int[] exclusionErrors;

        int taskCount;

        Map<Integer, TaskResult[]>taskResults = new HashMap<Integer, TaskResult[]>();
    }

    private static class TaskResult {
        boolean success = false;
        boolean executionError = false;
        boolean exclusionError = false;
        boolean inclusionError = false;
        int refExprSize = 0;
        boolean refExprTypeOnly = false;
        boolean refExprHasNeg = false;
        String refExprStr = "null";

        public String toString(){
            return success +"\t" + executionError + "\t" + exclusionError + "\t" + inclusionError + "\t" + refExprSize + "\t"+ refExprHasNeg + "\t" + refExprStr;
        }
    }

    private static class Task {
        List<URI> confusorsNew;
        List<URI> candidatesNew;
        List<URI> candidatesOld;
        List<URI> confusorsOld;
        URI referentNew;
        URI referentOld;
    }

    public static class Settings {
        static boolean isOrga;
        static boolean onlyDefined;
    }

    public static Data evalOne(int num, int all) throws RDFParseException, UnsupportedRDFormatException, IOException,
            RepositoryException {
        Data threadData = new Data();

        ReferringExpressionAlgorithm[] algorithms = new ReferringExpressionAlgorithm[] {
                new DaleReiterAlgorithm(TypePriorities.dbPediaPriorities, TypePriorities.dbPediaIgnored),
                new GardentAlgorithm(TypePriorities.dbPediaPriorities, TypePriorities.dbPediaIgnored),
                new GraphAlgorithm(TypePriorities.dbPediaPriorities, TypePriorities.dbPediaIgnored, 3000L) };

        FileInputStream inNew = new FileInputStream(Settings.isOrga ? "data/organization.tuples.new"
                : "data/people.tuples.new");
        Model mNew = Rio.parse(inNew, "http://localhost/", RDFFormat.NTRIPLES);

        Repository repNew = new SailRepository(new MemoryStore());
        repNew.initialize();
        ValueFactory fNew = repNew.getValueFactory();
        threadData.connNew = repNew.getConnection();
        threadData.connNew.add(mNew);

        FileInputStream inOld = new FileInputStream(Settings.isOrga ? "data/organization.tuples.old"
                : "data/people.tuples.old");
        Model mOld = Rio.parse(inOld, "http://localhost/", RDFFormat.NTRIPLES);

        Repository repOld = new SailRepository(new MemoryStore());
        repOld.initialize();
        ValueFactory fOld = repOld.getValueFactory();
        threadData.connOld = repOld.getConnection();
        threadData.connOld.add(mOld);

        Set<String>knownEntities = new HashSet<String>();
        if(Settings.onlyDefined){
            BufferedReader brOld = new BufferedReader(new FileReader(Settings.isOrga ? "data/organization.tuples.old"
                                                                   : "data/people.tuples.old"));
            String line = brOld.readLine();
            while(line!=null){
                String[]parts=line.split("\\s+");
                String uri = parts[0].substring(1,parts[0].length()-1);
                //System.out.println(uri);
                knownEntities.add(uri);
                line = brOld.readLine();
            }
            brOld.close();
        }

        BufferedReader br = new BufferedReader(new FileReader(Settings.isOrga ? "data/organization.tasks.linearized"
                : "data/people.tasks.linearized"));
        String line = br.readLine();

        double[] diceAccum = { 0.0, 0.0, 0.0 };
        int[] diceCount = { 0, 0, 0 };
        int[] solvedWithType = { 0, 0, 0 };
        int[] executionError = { 0, 0, 0 };
        int[] inclusionErrors = { 0, 0, 0 };
        int[] exclusionErrors = { 0, 0, 0 };

        int lineNum = 0;

        while (line != null) {
            if (lineNum % all == num) {
                String[] parts = line.split("\\s+");
                line = br.readLine();
                lineNum++;
                
                Task task = new Task();

                task.confusorsNew = new ArrayList<URI>(parts.length - 1);
                task.candidatesNew = new ArrayList<URI>(parts.length);
                task.candidatesOld = new ArrayList<URI>(parts.length);
                task.confusorsOld = new ArrayList<URI>(parts.length - 1);
                task.referentNew = fNew.createURI(parts[0]);
                task.referentOld = fOld.createURI(parts[0]);
                task.candidatesOld.add(task.referentOld);
                task.candidatesNew.add(task.referentNew);
                boolean allDefined = Settings.onlyDefined && knownEntities.contains(parts[0]);
                for (int i = 1; i < parts.length; i++) {
                    URI uriNew = fNew.createURI(parts[i]);
                    URI uriOld = fOld.createURI(parts[i]);
                    if(Settings.onlyDefined && allDefined && !knownEntities.contains(parts[i])) {
                        //System.out.println("Missing: " + parts[i]);
                        allDefined = false;
                        break;
                    }
                    task.confusorsNew.add(uriNew);
                    task.confusorsOld.add(uriOld);
                    task.candidatesNew.add(uriNew);
                    task.candidatesOld.add(uriOld);
                }

                if(Settings.onlyDefined && !allDefined)
                    continue;

                threadData.taskResults.put(lineNum, new TaskResult[algorithms.length]);

                for (int i = 0; i < algorithms.length; i++) {
                    ReferringExpressionAlgorithm algo = algorithms[i];
                    double thisDice = 0.0;
                    TaskResult taskResult = new TaskResult();
                    threadData.taskResults.get(lineNum)[i] = taskResult;
                    taskResult.success = true;
                    
                    Set<String> refExpSet = new HashSet<String>();
                    ReferringExpression refExp = null;
                    List<URI> newResult = null;
                    boolean inError = false;
                    try {
                        refExp = algo.resolve(task.referentOld, task.confusorsOld,
                                threadData.connOld);
                        for (Predicate predicate : refExp.predicates()) {
                            refExpSet.add(predicate.toString());
                        }
                        if (refExpSet.size() == 1
                                && refExp.predicates().get(0).getPredicate().getLocalName().equals("type")) {
                            // System.out.println(task.refExpTarget);
                            solvedWithType[i]++;
                            taskResult.refExprTypeOnly = true;
                        }
                        taskResult.refExprSize = refExp.predicates().size();
                        taskResult.refExprStr = refExp.toString().replaceAll("\\s+"," ").replaceAll("\n","\\n");
                        taskResult.refExprHasNeg = refExp.hasNegatives();

                        newResult = Resolver.resolve(refExp, task.candidatesNew, threadData.connNew);

                        if (newResult.contains(task.referentNew)) {
                            thisDice = 2.0 * 1 /* intersection */
                            / (1 /* original */+ newResult.size());
                            if (newResult.size() > 1) {
                                inclusionErrors[i]++;
                                inError = true;
                                taskResult.success = false;
                                taskResult.inclusionError = true;
                            }
                        } else {
                            exclusionErrors[i]++;
                            inError = true;
                            taskResult.success = false;
                            taskResult.exclusionError = true;
                            // thisDice stays 0.0, no intersection
                        }
                        diceAccum[i] += thisDice;
                        diceCount[i]++;
                        // } catch (ReferringExpressionException e) {
                    } catch (Throwable e) {
                        executionError[i]++;
                        inError = true;
                        taskResult.success = false;
                        taskResult.executionError = true;
                    }
                           
                    if(inError && Settings.onlyDefined){
                        PrintWriter pw = new PrintWriter(new FileWriter("/tmp/" + (Settings.isOrga ? "orga" : "people") + "." + algorithms[i].getClass().getSimpleName() +".errors", true));
                        String refExpStr = "null";
                        if(refExp != null)
                            refExpStr = refExp.toString().replaceAll("\\s+"," ").replaceAll("\n","\\n");
                        pw.println(task.referentOld + "\t" + task.confusorsOld +"\t" + newResult + "\t" + refExpStr);
                        pw.close();
                    }
                }
            }else{
                line = br.readLine();
                lineNum++;
            }
        }
        threadData.taskCount = lineNum;

        threadData.diceAccum = diceAccum;
        threadData.diceCount = diceCount;
        threadData.solvedWithType = solvedWithType;
        threadData.executionError = executionError;
        threadData.inclusionErrors = inclusionErrors;
        threadData.exclusionErrors = exclusionErrors;
        br.close();

        return threadData;
    }

    public static void eval() throws RDFParseException, UnsupportedRDFormatException, IOException, RepositoryException {

        final int cores = Runtime.getRuntime().availableProcessors();
        final AtomicInteger running = new AtomicInteger(cores);

        ReferringExpressionAlgorithm[] algorithms = new ReferringExpressionAlgorithm[] {
                new DaleReiterAlgorithm(TypePriorities.dbPediaPriorities, TypePriorities.dbPediaIgnored),
                new GardentAlgorithm(TypePriorities.dbPediaPriorities, TypePriorities.dbPediaIgnored),
                new GraphAlgorithm(TypePriorities.dbPediaPriorities, TypePriorities.dbPediaIgnored, 3000L) };

        final Data[] results = new Data[cores];

        for (int i = 0; i < cores; i++) {
            final int num = i;
            new Thread() {
                public void run() {
                    try {
                        results[num] = evalOne(num, cores);
                        running.decrementAndGet();
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                };

            }.start();
        }

        while (running.get() != 0) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

        double[] diceAccum = { 0.0, 0.0, 0.0 };
        int[] diceCount = { 0, 0, 0 };
        int[] solvedWithType = { 0, 0, 0 };
        int[] executionError = { 0, 0, 0 };
        int[] inclusionErrors = { 0, 0, 0 };
        int[] exclusionErrors = { 0, 0, 0 };

        PrintWriter pw =
            new PrintWriter(new FileWriter("/tmp/" +
                                           (Settings.isOrga ? "orga" : "people") + "." +
                                           (Settings.onlyDefined ? "defined" : "all") +
                                           ".results"));
        for (int c = 0; c < cores; c++) {
            for (int i = 0; i < algorithms.length; i++) {
                diceAccum[i] += results[c].diceAccum[i];
                diceCount[i] += results[c].diceCount[i];
                solvedWithType[i] += results[c].solvedWithType[i];
                executionError[i] += results[c].executionError[i];
                inclusionErrors[i] += results[c].inclusionErrors[i];
                exclusionErrors[i] += results[c].exclusionErrors[i];
            }
            for(Integer l:results[c].taskResults.keySet()){
                pw.print(l);
                for (int i = 0; i < algorithms.length; i++) 
                    pw.print("\t" + results[c].taskResults.get(l)[i]);
                pw.println();
            }
        }
        pw.close();

        int taskCount = results[0].taskCount;

        double[] dice = new double[algorithms.length];
        for (int i = 0; i < algorithms.length; i++) {
            dice[i] = diceAccum[i] / diceCount[i];

            System.out.println("Algorithm: " + algorithms[i]);
            System.out.println("Execution error: " + executionError[i] + " (" + (executionError[i] * 100.0 / taskCount)
                    + "%)");
            System.out.println("Successful runs: " + diceCount[i]);
            System.out.println("Dice: " + dice[i]);
            System.out.println("Inclusion errors: " + inclusionErrors[i]);
            System.out.println("Exclusion errors: " + exclusionErrors[i]);
            System.out.println("Solved with type: " + (solvedWithType[i] * 100.0 / taskCount) + "%");
            System.out.println();
        }

    }

    public static void main(String[] args) throws Exception {
        System.out.println("DEFINED\n\n");
        Settings.onlyDefined = true;
        System.out.println("People");
        Settings.isOrga = false;
        eval();
        System.out.println();
        System.out.println("Organizations");
        Settings.isOrga = true;
        eval();
        
        System.out.println("ALL\n\n");
        Settings.onlyDefined = false;
        System.out.println("People");
        Settings.isOrga = false;
        eval();
        System.out.println();
        System.out.println("Organizations");
        Settings.isOrga = true;
        eval();
    }
}
