#include <iomanip>
#include <algorithm>

#include "io/token_manager.h"
#include "baseline_status.h"

namespace baseline {

#define NIL -1

struct TreeNode {
    short first_child;
    short last_child;
    short sibling;
};

void Status::reset(int node_count) {
    score = 0;
    top_index = -1;
    rule_choices.assign(node_count, -1);
    stream_target.assign(node_count, -1);
    edge_states.clear();
    node_reached.reset();
}

void Status::printTree(std::ostream& os, const EdsGraph& graph) {
    static TreeNode tree[MAX_GRAPH_NODE_COUNT], root;
    static std::vector< std::pair< int, int > > stack;
    if (top_index == -1)
        return;
    int node_count = rule_choices.size();
    root = {NIL, NIL, NIL};
    std::fill_n(tree, node_count, root);

    for (int i = 0; i < node_count; ++i) {
        int parent = stream_target[i];
        auto& node = parent == NIL ? root : tree[parent];
        if (node.first_child == NIL) {
            node.first_child = i;
            node.last_child = i;
        } else {
            tree[node.last_child].sibling = i;
            node.last_child = i;
        }
    }

    stack.clear();
    stack.push_back({NIL, 0});
    while (!stack.empty()) {
        int index, depth;
        std::tie(index, depth) = stack.back();
        stack.pop_back();

        auto& node = index == NIL ? root : tree[index];
        if (depth > 0) {
            for (int i = 1; i < depth; ++i)
                os << "|  ";
            os << "|--";
        }
        if (index != NIL)
            os << index << '@' << graph.nodes[index] << '\n';
        else
            os << "*ROOT*" << '\n';

        if (node.sibling != NIL)
            stack.push_back({node.sibling, depth});
        if (node.first_child != NIL)
            stack.push_back({node.first_child, depth + 1});
    }
}
}
