import json

class Function(object):
    # required number of dependencies and inputs
    constrains = {
        'FindAll': [0, 0],                  # 
        'Find': [0, 1],                     # [], [entity_name]

        'FilterConcept': [1, 1],            # [entities], [concept_name]
        'FilterStr': [1, 2],                # [entities], [key, value]
        'FilterNum': [1, 3],                # [entities], [key, value, op]
        'FilterYear': [1, 3],               # [entities], [key, value, op]
        'FilterDate': [1, 3],               # [entities], [key, value, op]
        'QFilterStr': [1, 2],               # [facts], [qualifier_key, qualifier_value, op]
        'QFilterNum': [1, 3],               # [facts], [qualifier_key, qualifier_value, op]
        'QFilterYear': [1, 3],              # [facts], [qualifier_key, qualifier_value, op]
        'QFilterDate': [1, 3],              # [facts], [qualifier_key, qualifier_value, op]

        'Relate': [1, 2],                   # [entity], [predicate, direction]
        'And': [2, 0],                      # [entities_1, entities_2]
        'Or': [2, 0],                       # [entities_1, entities_2]

        'What': [1, 0],                     # [entity], []
        'Count': [1, 0],                    # [entities], []
        'SelectBetween': [2, 2],            # [entity_1, entity_2], [key, op]
        'SelectAmong': [1, 2],              # [entities], [key, op]

        'QueryAttr': [1, 1],                # [entity], [key]
        'QueryAttrUnderCondition': [1, 3],  # [entity], [key, qualifier_key, qualifier_value]
        'VerifyStr': [1, 1],                # [value], [value]
        'VerifyNum': [1, 2],                # [value], [value, op]
        'VerifyYear': [1, 2],               # [value], [value, op]
        'VerifyDate': [1, 2],               # [value], [value, op]

        'QueryRelation': [2, 0],            # [entity_1, entity_2], []

        'QueryAttrQualifier': [1, 3],       # [entity], [key, value, qualifier_key]
        'QueryRelationQualifier': [2, 2],   # [entity_1, entity_2], [predicate, qualifier_key]
    }

    def __init__(self, function, dependencies, inputs):
        assert len(dependencies) == self.constrains[function][0] and \
                len(inputs) == self.constrains[function][1]
        assert isinstance(dependencies, list) and isinstance(inputs, list)
        self.function = function
        self.dependencies = dependencies
        self.inputs = [str(i) for i in inputs]
        self.id = None # an integer to help convert dependencies to ints

    def dict(self):
        return {
            'function': self.function,
            'dependencies': self.dependencies,
            'inputs': self.inputs
            }

    def __str__(self):
        return json.dumps(self.dict())


from utils.misc import convert_program_list
class Question(object):
    def __init__(self, text='', program=[], sparql='', answer='', choices=[], info=None):
        self.text = text
        self.program = program
        self.sparql = sparql
        self.answer = answer
        self.choices = choices
        self.info = info

    def __str__(self):
        s = '---<Question>---:\n{}\n---<Porgram>---:\n{}\n---<Sparql>---:\n{}\n---<Answer>---:\n{}\n---<Choices>---:\n{}\n\n\n'.format(
            self.text,
            json.dumps([f.dict() for f in convert_program_list(self.program)], ensure_ascii=False, indent=2),
            self.sparql,
            str(self.answer),
            ';  '.join([str(c) for c in self.choices])
            )
        return s

    def dict(self):
        return {
            'text': self.text,
            'program': [f.dict() for f in convert_program_list(self.program)],
            'sparql': self.sparql,
            'answer': str(self.answer),
            'choices': [str(c) for c in self.choices]
        }

    def __eq__(self, other): # to remove duplicate questions
        return self.text == other.text

    def __hash__(self):
        return hash(self.text)


class ValueClass():
    def __init__(self, type, value, unit=None):
        """
        When type is
            - string, value is a str
            - quantity, value is a number and unit is required'
            - year, value is a int
            - date, value is a date object
        """
        self.type = type
        self.value = value
        self.unit = unit

    def isTime(self):
        return self.type in {'year', 'date'}

    def can_compare(self, other):
        if self.type == 'string':
            return other.type == 'string'
        elif self.type == 'quantity':
            # NOTE: for two quantity, they can compare only when they have the same unit
            return other.type == 'quantity' and other.unit == self.unit
        else:
            # year can compare with date
            return other.type == 'year' or other.type == 'date'

    def convert_to_year(self):
        # convert a date value to its year
        if self.type == 'year':
            return self
        elif self.type == 'date':
            return ValueClass('year', self.value.year)

    def contains(self, other):
        """
        check whether self contains other, which is different from __eq__ and the result is asymmetric
        used for conditions like whether 2001-01-01 in 2001, or whether 2001 in 2001-01-01
        """
        if self.type == 'year': # year can contain year and date
            other_value = other.value if other.type == 'year' else other.value.year
            return self.value == other_value
        elif self.type == 'date': # date can only contain date
            return other.type == 'date' and self.value == other.value
        else:
            raise Exception('not supported type: %s' % self.type)


    def __eq__(self, other):
        """
        2001 and 2001-01-01 is not equal
        """
        assert self.can_compare(other)
        return self.type == other.type and self.value == other.value

    def __lt__(self, other):
        """
        Comparison between a year and a date will convert them both to year
        """
        assert self.can_compare(other)
        if self.type == 'string':
            raise Exception('try to compare two string')
        elif self.type == 'quantity':
            return self.value < other.value
        elif self.type == 'year':
            other_value = other.value if other.type == 'year' else other.value.year
            return self.value < other_value
        elif self.type == 'date':
            if other.type == 'year':
                return self.value.year < other.value
            else:
                return self.value < other.value

    def __gt__(self, other):
        assert self.can_compare(other)
        if self.type == 'string':
            raise Exception('try to compare two string')
        elif self.type == 'quantity':
            return self.value > other.value
        elif self.type == 'year':
            other_value = other.value if other.type == 'year' else other.value.year
            return self.value > other_value
        elif self.type == 'date':
            if other.type == 'year':
                return self.value.year > other.value
            else:
                return self.value > other.value

    def __str__(self):
        if self.type == 'string':
            return self.value
        elif self.type == 'quantity':
            if self.value - int(self.value) < 1e-5:
                v = int(self.value)
            else:
                v = self.value
            return '{} {}'.format(v, self.unit) if self.unit != '1' else str(v)
        elif self.type == 'year':
            return str(self.value)
        elif self.type == 'date':
            return self.value.isoformat()

    def __hash__(self):
        return hash(str(self))
