import math

import numpy as np

from hedal.block import Block
from hedal.context import Context
from hedal.core.config import PackingType
from hedal.matrix.ops import block_ops as bop
from hedal.matrix.vector import HedalVector


def horizontal_sum(vec: HedalVector, direction: int, fill: bool = False, num_fill_cols: int = 0) -> HedalVector:
    """Horizontal sum of a vector, i.e. sum of each row. Will be a vector with a single block.

    Args:
        vec (HedalVector): Vector to be summed.
        direction (int): Direction of summation (0: left, 1: right).
        fill (bool, optional): Whether to fill the vector with result or not. Defaults to False.
        num_fill_cols (int, optional): Number of columns to be filled with results. Defaults to 0.
    
    Example:
        >>> v = HedalVector.from_ndarray(context, array=np.array([[1, 2, 3], [4, 5, 6]]))
        >>> vs = horizontal_sum(v, direction=0)
        >>> vs.to_ndarray()
        array([[6., 0., 0.], [15., 0., 0.]])

        when fill is True:
        >>> vs = horizontal_sum(v, direction=0, fill=True)
        >>> vs.to_ndarray()
        array([[6., 6., 6.], [15., 15., 15.]])

    Returns:
        HedalVector: Result vector of shape (vec.shape[0], vec.context.shape[1]).
    """
    res_block = Block.zeros(vec.context, vec.encrypted, type=vec.type)
    num_cols = vec.context.shape[1]

    for block in vec:
        block_sum = bop.sum(block, axis=1, direction=direction)
        res_block += block_sum

    mask_index = 0 if direction == 0 else num_cols - 1
    mask = Block.mask(vec.context, index=mask_index, axis=1, type=vec.type)
    res_block *= mask

    if fill:
        if num_fill_cols == 0:
            log_num_fill_cols = int(math.ceil(math.log2(num_cols)))
        else:
            log_num_fill_cols = int(math.ceil(math.log2(num_fill_cols)))
        if direction == 0:
            for idx in range(log_num_fill_cols):
                res_block += res_block >> (1 << idx)
        else:
            for idx in range(log_num_fill_cols):
                res_block += res_block << (1 << idx)

    res_vector = HedalVector(vec.context, block_list=[res_block], type=vec.type, encrypted=vec.encrypted)
    res_vector.shape = (vec.shape[0], vec.context.shape[1])
    return res_vector


def vertical_sum(vec: HedalVector, direction: int, fill: bool = False) -> HedalVector:
    """Vertical sum of a vector, i.e. sum of each column. Will be a vector with a same number of blocks as the original.

    Args:
        vec (HedalVector): vector to be summed.
        direction (int): Direction of summation (0: up, 1: down).
        fill (bool, optional): Whether to fill the vector with result or not. Defaults to False.
    
    Example:
        >>> v = HedalVector.from_ndarray(context, array=np.array([[1, 2, 3], [4, 5, 6]]))
        >>> vs = vertical_sum(v, direction=0)
        >>> vs.to_ndarray()
        array([[5., 7., 9.], [0., 0., 0.]])

        when fill is True:
        >>> vs = vertical_sum(v, direction=0, fill=True)
        >>> vs.to_ndarray()
        array([[5., 7., 9.], [5., 7., 9.]])

    Returns:
        HedalVector: Result vector of shape (vec.context.shape[0], vec.shape[1]).
    """
    res_block_list = []
    for block in vec:
        block_sum = bop.sum(block, axis=0, direction=direction)
        res_block_list.append(block_sum)
    if not fill:
        mask_index = 0 if direction == 0 else vec.context.shape[0] - 1
        mask = Block.mask(vec.context, index=mask_index, axis=0, type=vec.type)
        vec *= mask
    res_vector = HedalVector(vec.context, block_list=res_block_list, type=vec.type, encrypted=vec.encrypted)
    res_vector.shape = (vec.context.shape[0], vec.shape[1])

    return res_vector


def sum(vec: HedalVector, axis: int, direction: int, fill: bool = False) -> HedalVector:
    """Sum of a vector, i.e. sum of each row or column.
    See the functions horizontal_sum and vertical_sum for more details with examples.

    Args:
        vec (HedalVector): Vector to be summed.
        axis (int): Axis of summation (0: row, 1: column).
        direction (int): Direction of summation (0: left/up, 1: right/down).
        fill (bool, optional): Whether to fill the vector with result or not. Defaults to False.

    Raises:
        ValueError: If axis or direction is not 0 or 1.
        TypeError: If the type of the vector is not supported.

    Returns:
        HedalVector: Result vector.
    """
    if (axis not in (0, 1)) or (direction not in (0, 1)):
        raise ValueError(f"Axis({axis}) and direction({direction}) must be 0 or 1.")
    if vec.type != PackingType.MATRIX:
        raise TypeError("Invalid vector type")

    if axis == 0:
        return vertical_sum(vec, direction, fill)
    elif axis == 1:
        return horizontal_sum(vec, direction, fill)


def dot(v1: HedalVector, v2: HedalVector, fill: bool = False) -> HedalVector:
    return sum(v1 * v2, axis=1, direction=0, fill=fill)


def _off_diag_mask_col_tiled(context: Context, sub_size: int, offset: int):
    """Generate mask for vec_mul_tiled function.
    For example, if sub_size = 4 and offset = 1, then the mask is:
    [
        [0, 1, 0, 0],
        [0, 0, 1, 0],
        [0, 0, 0, 1],
        [1, 0, 0, 0],
        [0, 1, 0, 0],
        [0, 0, 1, 0],
        [0, 0, 0, 1],
        [1, 0, 0, 0],
        ...
    ]

    Args:
        context (Context): [description]
        sub_size (int): [description]
        offset (int): [description]
    """
    if context.shape[0] % sub_size != 0:
        raise ValueError(f"Invalid sub_size: {sub_size}")

    mask = np.zeros(context.shape)
    m = np.tile(np.roll(np.identity(sub_size), offset, axis=1), (context.shape[0] // sub_size, 1))
    mask[:, :sub_size] = m
    mask_block = Block.from_ndarray(context, mask)
    return mask_block


def _off_diag_mask_col_tiled_complex(context: Context, sub_size: int, offset: int):
    """Generate mask for vec_mul_tiled function with complex packing.
    It is defined as M_complex(k) = 0.5 * M_real(k) - 0.5 * i * M_real(k + sub_size / 2)
    where M_real(k) = _off_diag_mask_col_tiled(offset=k)

    Args:
        context (Context): [description]
        sub_size (int): [description]
        offset (int): [description]
    """
    ma = _off_diag_mask_col_tiled(context, sub_size, offset) * (1 / 2)
    mb = _off_diag_mask_col_tiled(context, sub_size, offset + sub_size // 2) * (1 / 2) * 1j
    return ma - mb


def vec_mul_col_tiled(v1: HedalVector, v2: HedalVector, complex: bool = False) -> HedalVector:
    """Product of two vectors, where the second vector v2 is assumed to be tiled along columns: if v2 has shape (c, b),
    then it actually contains a data as

    v2 = [
        [v2[0, 0], v2[0, 1], ..., v2[0, b-1]],
        [v2[1, 0], v2[1, 1], ..., v2[1, b-1]],
        ...,
        [v2[c-1, 0], v2[c-1, 1], ..., v2[c-1, b-1]],
        [v2[0, 0], v2[0, 1], ..., v2[0, b-1]],
        [v2[1, 0], v2[1, 1], ..., v2[1, b-1]],
        ...,
        [v2[c-1, 0], v2[c-1, 1], ..., v2[c-1, b-1]],
        ...,
    ]

    If v1 has shape (a, b) and v2 has shape (c, b), then the result has shape (a, c).

    Args:
        v1 (HedalVector): First vector.
        v2 (HedalVector): Second vector.
        complex (bool, optional): Whether input vectors have complex entries, or not. If False, then
            faster implementation will be used. Defaults to False.
    
    Returns:
        HedalVector: Result vector.
    """
    if v2.context.shape[0] % v2.shape[0] != 0:
        raise ValueError(
            f"v2.shape[0] ({v2.shape[0]}) must be a divisor of v2.context.shape[0] ({v2.context.shape[0]})."
        )

    res_encrypted = v1.encrypted or v2.encrypted
    if res_encrypted:
        res_vec = HedalVector(v1.context, shape=(v1.num_rows, v2.num_rows), type=v1.type, encrypted=res_encrypted)
        res_block = Block.zeros(v1.context, encrypted=res_encrypted, type=v1.type)
        if complex:
            for rot_idx in range(v2.num_rows):
                tmp = v1 * v2.rot_up(rot_idx)
                tmp = horizontal_sum(tmp, direction=0, fill=True, num_fill_cols=v2.num_rows).block_list[0]
                rot_id_mask = _off_diag_mask_col_tiled(v2.context, v2.num_rows, rot_idx)
                tmp *= rot_id_mask
                res_block += tmp
        else:  # faster multiplication for real vectors with less rotations
            v2_cplx = v2 + v2.rot_up(v2.num_rows // 2).i_mult()
            for rot_idx in range(v2.num_rows // 2):
                tmp = v1 * v2_cplx.rot_up(rot_idx)
                tmp = horizontal_sum(tmp, direction=0, fill=True, num_fill_cols=v2.num_rows).block_list[0]
                rot_id_mask = _off_diag_mask_col_tiled_complex(v2.context, v2.num_rows, rot_idx)
                tmp *= rot_id_mask
                res_block += tmp
            res_block += res_block.conjugate()
        res_vec.block_list = [res_block]
    else:
        v1_arr = v1.to_ndarray()
        v2_arr = v2.to_ndarray()
        res_arr = v1_arr @ v2_arr.T
        res_vec = HedalVector.from_ndarray(v1.context, res_arr)
    return res_vec


def _off_diag_mask_row_tiled(
    context: Context, sub_size: int, offset: int, num_cols: int, tile_col: bool = False
) -> HedalVector:
    """Generate mask for vec_mul_row_tiled function.
    For example, if sub_size = 4 and offset = 1, then the mask is:
    [
        [0, 0, 0, 1, 0, 0, 0, 1, ...],
        [1, 0, 0, 0, 1, 0, 0, 0, ...],
        [0, 1, 0, 0, 0, 1, 0, 0, ...],
        [0, 0, 1, 0, 0, 0, 1, 0, ...],
    ]
    When tile_col is True, then the mask will be tiled along columns:
    [
        [0, 0, 0, 1, 0, 0, 0, 1, ...],
        [1, 0, 0, 0, 1, 0, 0, 0, ...],
        [0, 1, 0, 0, 0, 1, 0, 0, ...],
        [0, 0, 1, 0, 0, 0, 1, 0, ...],

        [0, 0, 0, 1, 0, 0, 0, 1, ...],
        [1, 0, 0, 0, 1, 0, 0, 0, ...],
        [0, 1, 0, 0, 0, 1, 0, 0, ...],
        [0, 0, 1, 0, 0, 0, 1, 0, ...],
        ...
    ]

    Args:
        context (Context): Context of the vector.
        sub_size (int): Size of the sub-matrix. Should divide context.shape[1] evenly.
        offset (int): Offset of the sub-matrix.
        num_cols (int): Number of columns of the mask vector.
        tile_col (bool, optional): Whether to tile the mask along column too, or not. Defaults to False.

    Returns:
        HedalVector: Mask vector.
    """
    if context.shape[1] % sub_size != 0:
        raise ValueError(f"Invalid sub_size: {sub_size}")

    padded_num_cols = context.shape[1] * math.ceil(num_cols / context.shape[1])
    mask = np.zeros((context.shape[0], padded_num_cols))
    m = np.roll(np.identity(sub_size), offset, axis=0)
    m = np.tile(m, (1, padded_num_cols // sub_size))
    if tile_col:
        m = np.tile(m, (context.shape[0] // sub_size, 1))
        mask = m
    else:
        mask[:sub_size] = m
    mask[:, num_cols:] = 0
    mask_vec = HedalVector.from_ndarray(context, mask)
    return mask_vec


def _off_diag_mask_row_tiled_complex(
    context: Context, sub_size: int, offset: int, num_cols: int, tile_col: bool = False
) -> HedalVector:
    """Generate mask for vec_mul_row_tiled function with complex packing.
    It is defined as M_complex(k) = 0.5 * M_real(k) - 0.5 * i * M_real(k + sub_size / 2),
    where M_real(k) = _off_diag_mask_row_tiled(offset=k).

    Args:
        context (Context): Context of the vector.
        sub_size (int): Size of the sub-matrix. Should divide context.shape[1] evenly.
        offset (int): Offset of the sub-matrix.
        num_cols (int): Number of columns of the mask vector.
        tile_col (bool, optional): Whether to tile the mask along column too, or not. Defaults to False.

    Returns:
        HedalVector: Mask vector.
    """
    ma = _off_diag_mask_row_tiled(context, sub_size, offset, num_cols, tile_col) * (1 / 2)
    mb = _off_diag_mask_row_tiled(context, sub_size, offset + sub_size // 2, num_cols, tile_col) * (1 / 2) * 1j
    return ma - mb


def vec_mul_row_tiled(v1: HedalVector, v2: HedalVector, tile_col: bool = False, complex: bool = False) -> HedalVector:
    """Product of two vectors, where the first vector v1 is assumed to be tiled along rows: if v1 has shape (a, b),
    then it actually contains a data as

    v1 = [
        [v1[0, 0], v1[0, 1], ..., v1[0, b-1], v1[0, 0], v1[0, 1], ..., v1[0, b-1], ...],
        [v1[1, 0], v1[1, 1], ..., v1[1, b-1], v1[1, 0], v1[1, 1], ..., v1[1, b-1], ...],
        ...,
        [v1[a-1, 0], v1[al-1, 1], ..., v1[a-1, b-1], v1[a-1, 0], v1[a-1, 1], ..., v1[a-1, b-1], ...],
    ]

    If v1 has shape (a, b) and v1 has shape (a, c), then the result has shape (b, c).
    if tile_col is True, then the output will be tiled along columns, too.

    Args:
        v1 (HedalVector): First vector.
        v2 (HedalVector): Second vector.
        tile_col (bool, optional): Whether to tile the output along columns, or not. Defaults to False.
        complex (bool, optional): Whether input vectors have complex entries, or not. If False, then
            faster implementation will be used. Defaults to False.

    Returns:
        HedalVector: Result vector.
    """
    if v1.num_cols > v1.context.shape[1]:
        raise NotImplementedError(
            f"The case when v1.num_cols({v1.num_cols}) > v1.context.shape[1]({v1.context.shape[1]}) is not supported"
        )
    if v1.context.shape[1] % v1.num_cols != 0:
        raise ValueError(
            f"v1.num_cols ({v1.num_cols}) must be a divisor of v1.context.shape[1] ({v1.context.shape[1]})"
        )

    res_encrypted = v1.encrypted or v2.encrypted
    if res_encrypted:
        res_vec = HedalVector.zeros(v1.context, shape=(v1.num_cols, v2.num_cols), encrypted=res_encrypted)
        if complex:
            for rot_idx in range(v1.num_cols):
                tmp = v2.copy_memory()
                rot_v1 = v1.rot_left(rot_idx)
                for block in tmp:
                    block *= rot_v1[0]
                tmp = vertical_sum(tmp, direction=0, fill=True)
                rot_id_mask = _off_diag_mask_row_tiled(
                    v1.context, v1.num_cols, rot_idx, num_cols=v2.num_cols, tile_col=tile_col
                )
                tmp *= rot_id_mask
                res_vec += tmp
        else:  # faster multiplication for real vectors with less rotations
            v1_cplx = v1 + v1.rot_left(v1.num_cols // 2).i_mult()
            for rot_idx in range(v1.num_cols // 2):
                tmp = v2.copy_memory()
                rot_v1_cplx = v1_cplx.rot_left(rot_idx)
                for block in tmp:
                    block *= rot_v1_cplx[0]
                tmp = vertical_sum(tmp, direction=0, fill=True)
                rot_id_mask = _off_diag_mask_row_tiled_complex(
                    v1.context, v1.num_cols, rot_idx, num_cols=v2.num_cols, tile_col=tile_col
                )
                tmp *= rot_id_mask
                res_vec += tmp
            res_vec = res_vec + res_vec.conjugate()
    else:
        v1_arr = v1.to_ndarray()
        v2_arr = v2.to_ndarray()
        res_arr = v1_arr.T @ v2_arr
        if tile_col:
            res_arr = np.tile(res_arr, (v1.context.shape[0] // v1.shape[1], 1))
        res_vec = HedalVector.from_ndarray(v1.context, res_arr)
        res_vec.shape = (v1.shape[1], v2.shape[1])
    return res_vec


def sigmoid(vec: HedalVector, depth: int = 10) -> HedalVector:
    """Element-wise sigmoid function for vectors.

    Args:
        vec (HedalVector): Vector to be sigmoid-ed.
        depth (int, optional): Depth for sigmoid approximation. Defaults to 10.

    Returns:
        HedalVector: Result vector.
    """
    result = HedalVector(vec.context, shape=vec.shape, type=vec.type, encrypted=vec.encrypted)
    result.set_block_list([bop.sigmoid(block, depth) for block in vec])
    return result


def inverse(
    vec: HedalVector, one_slot: bool = False, greater_than_one: bool = True, inverse_num_iter: int = 20
) -> HedalVector:
    result = HedalVector(vec.context, shape=vec.shape, encrypted=vec.encrypted)
    result.set_block_list(
        [
            block.inverse(one_slot=one_slot, greater_than_one=greater_than_one, num_iter=inverse_num_iter)
            for block in vec
        ]
    )
    return result


def exp(vec: HedalVector, degree: int = 8) -> HedalVector:
    """Approximate exponential of a vector.
    It is approximated as exp(x) = (g(x) + 1)^8, where g(x) = exp(x/8) - 1 is approximated using taylor expension of given degree

    Args:
        vec (HedalVector): HedalVector to be exponentiated.
        degree (int, optional): Degree of Taylor approximation polynomial. Defaults to 8

    Returns:
        HedalVector: Approximated exponentiation result vector.
    """
    if vec.encrypted:
        scale_pow_param = 3
        vx = HedalVector.zeros(vec.context, shape=vec.shape)
        vy = vec.copy_memory() * (1 / (1 << scale_pow_param))
        for i in range(2, degree + 1):
            vz = vy + vec * (vy - vx) * (1 / (i * (1 << scale_pow_param)))
            if vz.need_bootstrap(2):
                vz.bootstrap()
            vx, vy = vy, vz
        vz += 1
        vz.bootstrap()  # FGb: level = 12
        for _ in range(scale_pow_param):
            vz *= vz
        return vz  # FGb: level = 9
    else:
        res_arr = vec.to_ndarray()
        res_arr = np.exp(res_arr)
        res_vec = HedalVector.from_ndarray(vec.context, res_arr)
        return res_vec


def exp_wide(vec: HedalVector, degree: int = 8, n: int = 3) -> Block:
    """Approximated exponential function.
    It is approximated as exp(x) = (g(x) + 1)^8, where g(x) = exp(x/8) - 1 is approximated using taylor expension of given degree
    Uses domain extension function to clip the exponentiation result.

    Args:
        block (Block): Block to compute exponential function.
        degree (int, optional): Degree of Taylor approximation polynomial. Defaults to 8
        n (int, optional): Number of iterations for domain extension function. Defaults to 3.

    Returns:
        Block: Approximated exponentiation result block.
    """
    if vec.encrypted:
        r = 10
        v = vec * (1 / r)
        bootstrap_m = int(np.ceil(2.25 ** n))
        for i in range(n, -1, -1):
            if v.need_bootstrap(4):
                v *= 1 / bootstrap_m
                v.bootstrap()
                v *= bootstrap_m
            v = v - (4 / (27 * (2.25 ** i))) * (v * v * v)
        v = exp(r * v, degree=degree)
        return v
    else:
        res_arr = vec.to_ndarray()
        res_arr = np.exp(res_arr)
        res_vec = HedalVector.from_ndarray(vec.context, res_arr)
        return res_vec


def softmax(vec: HedalVector, output_tiled: bool, exp_degree: int = 8, inverse_num_iter: int = 20) -> HedalVector:
    """Approximate row-wise softmax function.

    Args:
        vec (HedalVector): Vector to be softmaxed.
        output_tiled (bool): If True, the result is padded and tiled along rows.
            For example, if the real result is [a, b, c], then the padded & tiled result is [a, b, c, 0, a, b, c, 0, ...].
        exp_depth (int, optional): Degree of Taylor approximation for exponential. Defaults to 8.
        inverse_num_iter (int, optional): Number of iterations for inverse. Defaults to 20.

    Returns:
        HedalVector: Softmaxed vector. If output_tiled is True, then the shape of the result is (vec.shape[0], vec.context.shape[1]).
    """
    padded_num_cols = int(2 ** math.ceil(math.log2(vec.shape[1])))
    if vec.encrypted:
        # TODO: use first-slot subtraction only when the number of columns (number of classes) is 'small'
        # first_slot_mask = HedalVector.mask(vec.context, vec.shape, index=0, axis=1)
        # sub_block = (vec * first_slot_mask).block_list[0]
        # for rot_idx in range(min(math.ceil(math.log2(vec.shape[1])), int(math.log2(vec.context.shape[1])))):
        #     sub_block += sub_block >> (1 << rot_idx)

        # assumes that the result of exponential has enough level so that we don't need additional bootstrapping
        # this is ensured when using FGb parameter.
        exp_vec = exp(vec, degree=exp_degree)
        mask = HedalVector.from_ndarray(vec.context, array=np.ones(vec.shape))
        exp_vec *= mask

        exp_sum_vec = sum(exp_vec, axis=1, direction=0, fill=True)
        exp_sum_vec = inverse(exp_sum_vec, greater_than_one=True, inverse_num_iter=inverse_num_iter)
        res_vec = exp_vec * exp_sum_vec

        if output_tiled:
            rot_num = int(math.log2(vec.context.shape[1] // padded_num_cols))
            for i in range(rot_num):
                res_vec[0] += res_vec[0] >> ((1 << i) * padded_num_cols)
            res_vec.num_cols = padded_num_cols
        return res_vec
    else:
        arr = vec.to_ndarray()
        arr = arr - arr.max(axis=1, keepdims=True)
        arr = np.exp(arr)
        arr = arr / arr.sum(axis=1, keepdims=True)

        if output_tiled:
            arr = np.concatenate((arr, np.zeros((arr.shape[0], padded_num_cols - vec.shape[1]))), axis=1)
            arr = np.tile(arr, (1, vec.context.shape[1] // arr.shape[1]))
        res_vec = HedalVector.from_ndarray(vec.context, arr)
        res_vec.num_cols = padded_num_cols
        return res_vec


def softmax_wide(
    vec: HedalVector, output_tiled: bool, exp_degree: int = 8, inverse_num_iter: int = 20, n: int = 3,
) -> HedalVector:
    """Approximate row-wise softmax function.

    Args:
        vec (HedalVector): Vector to be softmaxed.
        output_tiled (bool): If True, the result is padded and tiled along rows.
            For example, if the real result is [a, b, c], then the padded & tiled result is [a, b, c, 0, a, b, c, 0, ...].
        exp_depth (int, optional): Degree of Taylor approximation for exponential. Defaults to 8.
        inverse_num_iter (int, optional): Number of iterations for inverse. Defaults to 20.

    Returns:
        HedalVector: Softmaxed vector. If output_tiled is True, then the shape of the result is (vec.shape[0], vec.context.shape[1]).
    """
    padded_num_cols = int(2 ** math.ceil(math.log2(vec.shape[1])))
    if vec.encrypted:
        # TODO: use first-slot subtraction only when the number of columns (number of classes) is 'small'
        # first_slot_mask = HedalVector.mask(vec.context, vec.shape, index=0, axis=1)
        # sub_block = (vec * first_slot_mask).block_list[0]
        # for rot_idx in range(min(math.ceil(math.log2(vec.shape[1])), int(math.log2(vec.context.shape[1])))):
        #     sub_block += sub_block >> (1 << rot_idx)

        # assumes that the result of exponential has enough level so that we don't need additional bootstrapping
        # this is ensured when using FGb parameter.
        exp_vec = exp_wide(vec, degree=exp_degree, n=n)
        mask = HedalVector.from_ndarray(vec.context, array=np.ones(vec.shape))
        exp_vec *= mask

        exp_sum_vec = sum(exp_vec, axis=1, direction=0, fill=True)
        exp_sum_vec = inverse(exp_sum_vec, greater_than_one=True, inverse_num_iter=inverse_num_iter)
        res_vec = exp_vec * exp_sum_vec

        if output_tiled:
            rot_num = int(math.log2(vec.context.shape[1] // padded_num_cols))
            for i in range(rot_num):
                res_vec[0] += res_vec[0] >> ((1 << i) * padded_num_cols)
            res_vec.num_cols = padded_num_cols
        return res_vec
    else:
        return softmax(vec, output_tiled=output_tiled, exp_degree=exp_degree, inverse_num_iter=inverse_num_iter)
